# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
from subprocess import Popen


def make_command(game_path, binary, excluded_dlcs, mods):
    """Make a list of components of a shell command.  Every space-separated
    item gets its own entry.

    The list starts with a full path to the binary, followed by entries for
    excluding DLCs and/or enabling mods.

    :param game_path: Absolute path to the root directory of the game.
    :type game_path: str.
    :param binary: Name of the binary.
    :type binary: str.
    :param excluded_dlcs: List of names of DLCs that should be disabled.
    :type excluded_dlcs: list.
    :param mods: List of names of mods that should be enabled.
    :type mods: list.
    :returns: List of typically space-separated items of a shell command.
    :rtype: list.
    """
    command = [os.path.join(game_path, binary)]
    command.append("-skiplauncher")
    for dlc in excluded_dlcs:
        command.append("-exclude_dlc={}".format(os.path.join("dlc", dlc)))
    for mod in mods:
        command.append("-mod={}".format(os.path.join("mod", mod)))

    return command


def execute_command(command):
    """Execute a shell command and return the resulting ``Popen`` object.

    :param command: Command that is forwarded as shell command.
    :type command: str or list.
    :returns: A process.
    :rtype: Popen.
    """
    return Popen(command)
