# Copyright (C) 2008, Charles Wang <charlesw1234@163.com>
# Author: Charles Wang <charlesw1234@163.com>

import os.path
import string
import sys

from pypi2pkgsys.pypi2packages import config
from pypi2pkgsys.PackageSystem import PackageSystem
from pypi2pkgsys.utils import *

pypi_dir = 'dev-python'

ebuild_format = """# This file is generated by %(self)s automatically.

inherit distutils

MY_PN=%(name)s
MY_PF=%(package_file)s
MY_PD=%(package_dir)s

DESCRIPTION="%(desc)s"
HOMEPAGE="%(url)s"
SRC_URI="http://what/${MY_PF}"
LICENSE="%(license)s"
KEYWORDS="~alpha ~amd64 ~hppa ~ia64 ~ppc ~ppc64 ~sparc ~x86"
SLOT="0"
IUSE="%(iuse)s"

RDEPEND="%(rdepend)s"
DEPEND="${RDEPEND}
        %(depend)s"

S=${WORKDIR}/${MY_PD}

src_unpack() {
        distutils_src_unpack
        %(patchlist)s
}

src_compile() {
        distutils_src_compile
        if use doc ; then
                PYTHONPATH=. "${python}" setup.py pudge || die "generating doc failed."
        fi
}

src_install() {
        distutils_src_install
        use doc && dohtml -r docs/html/*
}
"""

version_map = {}
for name, value in config.items('portage-version-map'):
    version_map[name] = value

class PkgSysPortage(PackageSystem):
    def __init__(self):
        PackageSystem.__init__(self)

    def InitializeOptions(self, options):
        options['--portage-distfiles'] = '/usr/portage/distfiles'
        options['--portage-dir'] = os.path.join('/usr/local/portage',
                                                pypi_dir)
        return options

    def FinalizeOptions(self, options):
        map(lambda diropt: ensure_dir(options[diropt]),
            ['--portage-distfiles', '--portage-dir'])
        return options

    def GenPackage(self, args, options):
        def NameConvert(pyname):
            return pyname.lower()
        def VersionConvert(pyname, pyversion):
            fullname = '%s-%s' % (pyname, pyversion)
            if fullname in version_map: return version_map[fullname]
            ver = ''
            while pyversion != '':
                if pyversion[0].isdigit() or pyversion[0] == '.':
                    ver = ver + pyversion[0]
                    pyversion = pyversion[1:]
                else:
                    ver = ver + '_' + pyversion
                    break
            return ver
        def DepConvert(pydep):
            if '>=' not in pydep:
                return '%s/%s' % (pypi_dir, NameConvert(pydep))
            idx = pydep.index('>=')
            name = pydep[:idx].strip(); version = pydep[idx + 2:].strip()
            return '>=%s/%s-%s' % (pypi_dir,
                                   NameConvert(name),
                                   VersionConvert(name, version))

        eb_args = {}
        eb_args['self'] = sys.argv[0]
        eb_args['name'] = args['name']
        eb_args['package_file'] = args['package_file']
        eb_args['package_dir'] = args['package_dir']
        eb_args['desc'] = args['description']
        eb_args['url'] = args['url']
        eb_args['license'] = args['license']

        iuse_arr = ['doc'] + args['extras_require'].keys()
        eb_args['iuse'] = string.join(iuse_arr)

        rdepend = map(lambda dep: DepConvert(dep),
                      args['install_requires'])
        eb_args['rdepend'] = string.join(rdepend, '\n\t')

        ereq = {}
        for k in args['extras_require']:
            if args['extras_require'][k] == []: continue
            ereq[k] = args['extras_require'][k]
        depend = []
        if args['name'] != 'setuptools':
            depend.append('dev-python/setuptools')
        depend.append('doc? ( dev-python/pudge dev-python/buildutils )')
        depend.extend(map(lambda extra: '%s? ( %s )' % \
                              (extra, string.join(map(lambda edep:
                                                          DepConvert(edep),
                                                      ereq[extra]))),
                          ereq.keys()))
        eb_args['depend'] = string.join(depend, '\n\t')
        eb_args['patchlist'] = \
            string.join(map(lambda p: 'epatch "${FILESDIR}/%s"' % p,
                            args['package_patches']), '\n        ')

        if 'config_rev' not in args:
            ebuild_fn = '%s-%s.ebuild' % \
                (NameConvert(args['name']),
                 VersionConvert(args['name'], args['version']))
        else:
            ebuild_fn = '%s-%s-r%s.ebuild' % \
                (NameConvert(args['name']),
                 VersionConvert(args['name'], args['version']),
                 args['config_rev'])
        ebuild_dir = os.path.join(options['--portage-dir'],
                                  NameConvert(args['name']))
        ensure_dir(ebuild_dir)
        ebuild_path = os.path.join(ebuild_dir, ebuild_fn)
        updated = False
        print 'Writing %s' % ebuild_path
        if smart_write(ebuild_path, ebuild_format % eb_args):
            updated = True
        if smart_symlink(args['package_path'],
                         os.path.join(options['--portage-distfiles'],
                                      args['package_file'])):
            updated = True
        if args['package_patches'] != []:
            ebuild_dir_files = os.path.join(ebuild_dir, 'files')
            ensure_dir(ebuild_dir_files)
            for p in args['package_patches']:
                if smart_symlink(os.path.join(os.path.dirname(__file__),
                                              'patches', p),
                                 os.path.join(ebuild_dir_files, p)):
                    updated = True
                
        os.system('ebuild %s digest' % (ebuild_path))

        deps = []
        if updated:
            deps = uniq_extend(deps, args['install_requires'])
            for k in args['extras_require'].keys():
                deps = uniq_extend(deps, args['extras_require'][k])
        return (updated, deps)
