import time
from collections import OrderedDict
from datetime import datetime
from threading import Timer
from functools import wraps
from dateutil.relativedelta import relativedelta
from dateutil.rrule import rrule, YEARLY, MONTHLY,\
		DAILY, HOURLY, MINUTELY, SECONDLY
from dateutil.parser import parse
from pyriodic import settings as s
from pyriodic.settings import NEW, WAITING, COMPLETE,\
		DONE, FAILED, RECURRING, DELAYED

class Threader(Timer):
	def __init__(self, job):
		self.job = job
		super().__init__(self.job.seconds_to_run, self.job.function,
				self.job.args, self.job.kwargs)

	def run(self):
		if self.interval >= 0:
			self.job.status = WAITING
			try:
				self.finished.wait(self.interval)
			except OverflowError:
				while self.interval > s.sleep_max:
					self.interval -= s.sleep_max
					self.finished.wait(s.sleep_max)
				self.finished.wait(self.interval)
			if not self.finished.is_set():
				try:
					result = self.function(*self.args, **self.kwargs)
					if result and result is not self.job.function:
						self.job.results[datetime.now()] = result
					self.finished.set()
					self.job.status = COMPLETE
					self.job.run_count += 1
				except Exception as e:
					self.job.status = FAILED
					self.job.error_count += 1
					self.job.log.error('Error in', self.job.name, 'error count at',
							self.job.error_count, 'Traceback:', e)
					raise

class Job(object):
	run_count = 0
	error_count = 0
	seconds_to_run = None
	next_run_time = None
	thread = None

	def __init__(self, func, **settings):
		self.function = func
		self.id = id(func)
		self.name = settings['name'] or func.__name__
		self.args = settings['args'] if settings['args'] is not None else []
		self.kwargs = settings['kwargs'] if settings['kwargs'] is not None else {}
		self.job_type = settings['job_type']
		self.status = settings['status'] or s.status
		self.retrys = settings['retrys'] or s.retrys
		self.retry_delay = settings['retry_delay'] or s.retry_delay
		self.how_many = settings['how_many'] or s.how_many
		self.log = settings['log'] or s.LogStub
		self.results = OrderedDict()

class RecuringJob(Job):
	def __init__(self, func, **settings):
		super().__init__(func,
		        name=settings['name'],
		        args=settings['args'],
		        kwargs=settings['kwargs'],
		        job_type=RECURRING,
		        status=settings['status'] or s.status,
		        retrys=settings['retrys'] or s.retrys,
		        retry_delay=settings['retry_delay'] or s.retry_delay,
		        how_many=settings['how_many'] or s.how_many
		)
		self.interval = settings['interval']
		self.days_of_week = settings['days_of_week']
		self.start_datetime = settings['start_datetime']

	def find_next_time(self):
		time_delta = self.start_datetime - datetime.now()
		if time_delta.days < 0:
			if self.interval == SECONDLY:
				self.start_datetime = datetime.now()
			elif self.interval == MINUTELY:
				self.start_datetime = self.start_datetime.replace(
					datetime.now().year,
					datetime.now().month,
					datetime.now().day,
					datetime.now().hour,
					datetime.now().minute,
					microsecond=0
				)
			elif self.interval == HOURLY:
				self.start_datetime = self.start_datetime.replace(
					datetime.now().year,
					datetime.now().month,
					datetime.now().day,
					datetime.now().hour,
					microsecond=0
				)
			elif self.interval == DAILY:
				self.start_datetime = self.start_datetime.replace(
					datetime.now().year,
					datetime.now().month,
					datetime.now().day,
					microsecond=0
				)
			elif self.interval == MONTHLY:
				self.start_datetime = self.start_datetime.replace(
					datetime.now().year,
					datetime.now().month,
					microsecond=0
				)
			elif self.interval == YEARLY:
				self.start_datetime = self.start_datetime.replace(
					datetime.now().year,
					microsecond=0
				)
		runs = list(rrule(self.interval, dtstart=self.start_datetime, count=2))
		self.seconds_to_run = (runs[0] - datetime.now()).total_seconds()
		self.next_run_time = runs[1]
		count = 1
		while True:
			if self.seconds_to_run > 0:
				break
			runs = list(rrule(self.interval, dtstart=self.start_datetime, count=(count + 2)))
			self.seconds_to_run = (runs[count] - datetime.now()).total_seconds()
			self.next_run_time = runs[count + 1]
			count += 1

	def schedule_next(self):
		if self.error_count < self.retrys:
			if self.run_count < self.how_many:
				if self.status is not WAITING:
					if self.status is NEW:
						self.find_next_time()
					elif self.status is FAILED:
						self.seconds_to_run = self.retry_delay
					else:
						self.seconds_to_run = (self.next_run_time - datetime.now()).total_seconds()
					if self.seconds_to_run > 0:
						self.thread = Threader(self)
						self.thread.start()
					if self.status is not NEW:
						self.find_next_time()
			else:
				self.status = DONE
		else:
			self.status = DONE

class DelayedJob(Job):
	def __init__(self, func, **settings):
		super().__init__(func,
		        name=settings['name'],
		        args=settings['args'],
		        kwargs=settings['kwargs'],
		        job_type=DELAYED,
		        status=settings['status'] or s.status,
		        retrys=settings['retrys'] or s.retrys,
		        retry_delay=settings['retry_delay'] or s.retry_delay,
		        how_many=settings['how_many'] or s.how_many
		)
		self.time_delta = settings['time_delta']
		self.years = settings['years']
		self.months = settings['months']
		self.days = settings['days']
		self.hours = settings['hours']
		self.minutes = settings['minutes']
		self.seconds = settings['seconds']

	def schedule_next(self):
		if self.error_count < self.retrys:
			if self.run_count < self.how_many:
				if self.status is not WAITING:
					if self.status is FAILED:
						self.seconds_to_run = self.retry_delay
					else:
						self.time_delta = datetime.now() + relativedelta(
								years=self.years,
								months=self.months,
								days=self.days,
								hours=self.hours,
								minutes=self.minutes,
								seconds=self.seconds,
								microseconds=0) - datetime.now()
						# The line above may seem strange but it creates a
						# timedelta object; datetime + relativedelta = new_datetime
						# new_datetime - datetime = timedelta
						self.seconds_to_run = self.time_delta.total_seconds()
					self.thread = Threader(self)
					self.thread.start()
			else:
				self.status = DONE
		else:
			self.status = DONE

class Scheduler:
	job_queue = OrderedDict()

	def __init__(self):
		def event_loop():
			while True:
				start = time.clock()
				for key in self.job_queue:
					if self.job_queue[key].status is DONE:
						del self.job_queue[key]
					elif self.job_queue[key].status is not WAITING:
						self.job_queue[key].schedule_next()
				wait = s.loop_wait - (time.clock() - start)
				if wait  >= 0:
					time.sleep(wait)
		Timer(0, event_loop).start()

	def schedule_recuring_job(self,
	        interval=DAILY,
	        start_date=s.start_date,
	        start_time=s.start_time,
	        days_of_week=None,
	        log=None,
	        how_many=s.how_many,
	        retrys=s.retrys,
	        retry_delay=s.retry_delay,
	        name=None):
		def wrap(f):
			def wrapped(*args, **kwargs):
				Scheduler.job_queue[name or self.func_name(f)] = RecuringJob(f,
				        args=args,
				        kwargs=kwargs,
				        interval=interval,
				        log=log,
				        start_datetime=parse('%s %s' % (start_date, start_time),
				                yearfirst=True,
				                ignoretz=True),
				        days_of_week=days_of_week,
				        how_many=how_many,
				        retrys=retrys,
				        retry_delay=retry_delay,
				        name=name
				)
			return wraps(f)(wrapped)
		return wrap

	def add_recuring_job(self, f,
	        args=None,
	        kwargs=None,
	        interval=DAILY,
	        start_date=s.start_date,
	        log=None,
	        start_time=s.start_time,
	        days_of_week=None,
	        how_many=s.how_many,
	        retrys=s.retrys,
	        retry_delay=s.retry_delay,
	        name=None):
		self.job_queue[name or self.func_name(f)] = RecuringJob(f,
		        args=args,
		        kwargs=kwargs,
		        interval=interval,
		        log=log,
				start_datetime=parse('%s %s' % (start_date, start_time),
						yearfirst=True,
						ignoretz=True),
				days_of_week=days_of_week,
				how_many=how_many,
				retrys=retrys,
				retry_delay=retry_delay,
				name=name
		)

	def schedule_delayed_job(self,
	        years=0,
	        months=0,
	        days=0,
	        hours=0,
	        minutes=0,
	        seconds=0,
	        how_many=s.how_many,
	        retrys=s.retrys,
	        retry_delay=s.retry_delay,
	        name=None,
	        log=None):
		def wrap(f):
			def wrapped(*args, **kwargs):
				Scheduler.job_queue[name or self.func_name(f)] = DelayedJob(f,
				        args=args,
						kwargs=kwargs,
						years=years,
						months=months,
						log=log,
				        days=days,
				        hours=hours,
				        minutes=minutes,
				        seconds=seconds,
						how_many=how_many,
						retrys=retrys,
						retry_delay=retry_delay,
						name=name)
			return wraps(f)(wrapped)
		return wrap

	def add_delayed_job(self, f,
	        args=None,
	        kwargs=None,
	        years=0,
	        months=0,
	        days=0,
	        hours=0,
	        minutes=0,
	        seconds=0,
	        how_many=s.how_many,
	        retrys=s.retrys,
	        retry_delay=s.retry_delay,
	        name=None,
	        log=None):
		self.job_queue[name or self.func_name(f)] = DelayedJob(f,
		        args=args,
		        kwargs=kwargs,
		        years=years,
		        log=log,
		        months=months,
		        days=days,
		        hours=hours,
		        minutes=minutes,
		        seconds=seconds,
		        how_many=how_many,
		        retrys=retrys,
		        retry_delay=retry_delay,
		        name=name
		)

	def func_name(self, func):
		if type(func) is not str:
			try:
				func = func.__name__
			except AttributeError:
				raise NameError('% does not seem to be a valid function or name.' % func)
		return func

	def remove_job(self, func):
		func = self.func_name(func)
		try:
			try:
				self.job_queue[func].thread.finished.set()
			except AttributeError:
				pass
			del self.job_queue[func]
			print(func, 'was removed.')
		except KeyError:
			print('The job', func, 'was not in the Queue')

	def get_results(self, func, run_date=None, run_time=None):
		func = self.func_name(func)
		return self.job_queue[func].results

	# Is primarily being used for debuging
	def get_job(self, func):
		func = self.func_name(func)
		return self.job_queue[func]