#-*- coding: utf-8 -*-

# Copyright 2013 Juca Crispim <jucacrispim@gmail.com>

# This file is part of pyrocumulus.

# pyrocumulus is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pyrocumulus is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pyrocumulus.  If not, see <http://www.gnu.org/licenses/>.


import os
import sys
import argparse
import importlib
from pyrocumulus.exceptions import(PyrocumulusCommandNotFound,
                                   PyrocumulusException)


class BaseCommand:
    """
    Base class for all commands. Your subclass must implement
    run().

    If your command accept command line arguments, just add they
    to the user_options list. Each command in this list is a
    dictionary, containing two keys: 'args', and 'kwargs'. Like this:

    [{'args': ('opt1',)
     'kwargs': {'default': True}}, ...]

    'args' and 'kwargs' are arguments passed to ArgumentParser.add_argument
    from argparse module. For details take a look here

    http://docs.python.org/3.3/howto/argparse.html#id1

    and here

    http://docs.python.org/3.3/library/argparse.html#module-argparse
    """

    description = None
    # --settings=settings.module
    default_options = [{'args': ('--settings',),
                        'kwargs': {'default': None,
                                   'help': 'settings module to use',
                                   'nargs': '?'}}]
    user_options = []

    def __init__(self):
        self.name = self.get_command_name()
        self.parser = argparse.ArgumentParser()
        self.add_args()
        self.parse_args()
        self.setenv()

    def add_args(self):
        """
        Add the arguments in options
        """
        # first, adding the command itself, since the command is passed
        # as a param to pyromanager or pyrocumulus
        cmd_description = self.description or '%s command' % self.name
        self.parser.add_argument(self.name, help=cmd_description)

        self.user_options += self.default_options
        for option in self.user_options:
            self.parser.add_argument(*option['args'], **option['kwargs'])

    def parse_args(self):
        """
        Parse the args from the command line
        """
        options = self.parser.parse_args()
        self.settings = None
        args = [opt for opt in dir(options) if not opt.startswith('_')]
        for arg in args:
            setattr(self, arg, getattr(options, arg))

    def run(self):
        """
        Execute your action. Your command must implement this
        """
        raise NotImplementedError

    def get_command_name(self):
        name = self.__class__.__module__.split('.')[-1]
        return name

    def setenv(self):
        """
        Set the env variable PYROCUMULUS_SETTINGS_MODULE
        using the value passed on --setting option
        """
        if self.settings:
            os.environ['PYROCUMULUS_SETTINGS_MODULE'] = self.settings


def get_command(name):
    """
    Retuns a command based on `name`. The name of the command will be
    the file name, without the extension. So, if we have a mycommand.py
    file, the command name will be mycommand.
    """

    package_name = 'pyrocumulus.commands'
    module = importlib.import_module('.'+name, package=package_name)
    names = [n for n in dir(module) if not n.startswith('_')]
    for cname in names:
        cmd_class = getattr(module, cname)
        if type(cmd_class) == type(BaseCommand) and \
           issubclass(cmd_class, BaseCommand) and \
           cmd_class != BaseCommand:

            return cmd_class

    raise PyrocumulusCommandNotFound(name)


def get_command_name():
    """
    Returns the pyrocumulus command name based on the
    command line
    """
    try:
        if 'python' in sys.argv[0]:
            cmdname = sys.argv[2]
        else:
            cmdname = sys.argv[1]
    except IndexError:
        raise PyrocumulusException('Invalid command line')

    return cmdname


def run_command():
    """
    runs a pyrocumulus command based on the command line
    """
    command_name = get_command_name()
    command = get_command(command_name)()
    command.run()
