import scipy as sp
import copy
import pysep
import pysep.prop_db as db

class component(pysep.base):

    def __init__(self,name='',**props):
        super().__init__(name)
        #Fetch pure component from prop_db
        try:
            self.constants = pysep.catalogue.get(name=name)
        except:
            print('Requested component not found, returning a blank component')
            self.constants = {'Pc'    : sp.nan,
                              'Tc'    : sp.nan,
                              'A'     : sp.nan,
                              'B'     : sp.nan,
                              'C'     : sp.nan,
                              'Cp'    : sp.nan,
                              'Hvap'  : sp.nan,
                              }
        
    def __repr__(self):
        header = '-'*50
        print(header)
        print('{h:15s} {c:3s} {n}'.format(h='Component Name', c=' : ', n=self.name))
        print(header)
        for item in self.constants.keys():
            print('{h:15s} {c:3s} {n}'.format(h=item, c=' : ', n=self.constants[item]))
        print(header)
        return ''
        
    def Pvap(self,T):
        A = self.constants['A']
        B = self.constants['B']
        C = self.constants['C']
        P = 10**(A - B/(T + C))*101325
        return P
        
    def Tbub(self,P):
        A = self.constants['A']
        B = self.constants['B']
        C = self.constants['C']
        T = - B/(sp.log10(P/101325) - A) - C
        return T
        
    def H(self,T):
        Cp = self.Cp(T=T)
        Tref = 273.13
        H = Cp*(T - Tref)
        return H
        
    def Cp(self,T=298.15):
        Cp = self.constants['Cp']
        return Cp
        
class stream(pysep.base,dict):

    def __init__(self,name='',comps=[],concs=[],**kwargs):
        super().__init__(name)
        #Define default stream properties
        stream_props = ['molar_flow',
                        'temperature',
                        'pressure',
                        ]
        stream_data = {}
        [stream_data.update({i:sp.nan}) for i in stream_props]

        #Over write default property values (nan) with any supplied values
        stream_data.update(kwargs.items())
        
        #Initialize properties dictionary
        self.update(stream_data)
        
        #Initialize object-specific attributes
        self._comps = []
        self['components'] = {}
        self['components']['mole_fraction'] = {}
        
        #Assign any components and compositions supplied in init
        if len(comps) > 0:
            for item in zip(comps,concs):
                self.add_component(comp=item[0],conc=item[1])
            
    def add_component(self,comp,conc=0.0,**kwargs):
        r'''
        '''
        self['components']['mole_fraction'][comp.name] = conc
        self._comps.append(comp)
        
    def clone(self,mode='reset'):
        temp = copy.deepcopy(self)
        temp.name = ''
        if mode == 'reset':
            for item in temp.keys():
                temp[item] = sp.nan
            temp['components'] = {}
            for item in temp['components'].keys():
                temp['components'][item] = {}
            temp._comps = []
        elif mode == 'full':
            pass
        return temp
        
    def __repr__(self):
        header = '-'*50
        print(header)
        print('{h:15s} {c:3s} {n}'.format(h='Stream Name', c=' : ', n=self.name))
        print(header)
        prop_list = list(self.keys())
        prop_list.remove('components')
        for item in prop_list:
            print('{h:15s} {c:3s} {n}'.format(h=item, c=' : ', n=self[item]))
        print(header)
        print('{h:15s} {c:3s}'.format(h='Components', c=' : '),end='')
        for item in self._comps:
            print('{n:15s}'.format(n=item.name),end='')
        print('\n',end='')
        print(header)
        for item in self['components'].keys():
            #Print property name
            print('{h:15s} {c:3s}'.format(h=item, c=' : '),end='')
            for comp in self._comps:
                #Print property value for each component
                print('{n:<15f}'.format(n=self['components'][item][comp.name]),end='')
            
        return ''
        
        
        
class catalogue():
    @staticmethod
    def get(name):
        consts = getattr(db,name)()
        return consts

        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        