#
#------------------------------------------------------------------------------
# Copyright (c) 2013-2014, Christian Therien
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#------------------------------------------------------------------------------
#
# utests.py - This file is part of the PySptools package.
#

import unittest
import numpy as np
from pysptools.classifiers import SAM, SID, NormXCorr
from types import *

# B suffix == bad
# G suffix == good

class TestClassifierError(unittest.TestCase):

    def setUp(self):
        pass


    def runTest(self):
        print('==> runTest: TestClassifierError')
        self.test_not_ndarray()
        self.test_wrong_input_dim()
        self.test_threshold_wrong_type()
        self.test_threshold_wrong_length()
        self.test_threshold_wrong_value()


    def test_not_ndarray(self):
        data_G = np.zeros((2,2,2,2))
        data_B = [1,2]
        E_G = np.zeros((2,2))
        E_B = 123

        cls = SAM()
        # err1
        with self.assertRaises(RuntimeError):
            cls.classify(data_B, E_G)
        # err2
        with self.assertRaises(RuntimeError):
            cls.classify(data_G, E_B)

        cls = SID()
        # err1
        with self.assertRaises(RuntimeError):
            cls.classify(data_B, E_G)
        # err2
        with self.assertRaises(RuntimeError):
            cls.classify(data_G, E_B)

        cls = NormXCorr()
        # err1
        with self.assertRaises(RuntimeError):
            cls.classify(data_B, E_G)
        # err2
        with self.assertRaises(RuntimeError):
            cls.classify(data_G, E_B)


    def test_wrong_input_dim(self):
        data_B = np.zeros((2,2,2,2))
        data_G = np.zeros((3,3,3))
        E_G = np.zeros((2,2))
        E_B = np.zeros((1,2,1,2))

        cls = SAM()
        # err3
        with self.assertRaises(RuntimeError):
            cls.classify(data_B, E_G)
        # err4
        with self.assertRaises(RuntimeError):
            cls.classify(data_G, E_B)

        cls = SID()
        # err3
        with self.assertRaises(RuntimeError):
            cls.classify(data_B, E_G)
        # err4
        with self.assertRaises(RuntimeError):
            cls.classify(data_G, E_B)

        cls = NormXCorr()
        # err3
        with self.assertRaises(RuntimeError):
            cls.classify(data_B, E_G)
        # err4
        with self.assertRaises(RuntimeError):
            cls.classify(data_G, E_B)


    def test_threshold_wrong_type(self):
        data_G = np.zeros((3,3,3))
        E_G = np.zeros((2,2))
        threshold_B = [1,2]

        cls = SAM()
        # err5
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold_B)

        cls = SID()
        # err5
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold_B)

        cls = NormXCorr()
        # err5
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold_B)


    def test_threshold_wrong_length(self):
        data_G = np.zeros((3,3,3))
        E_G = np.zeros((2,2))
        threshold_B = [1,2,3]

        cls = SAM()
        # err6
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold_B)

        cls = SID()
        # err6
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold_B)

        cls = NormXCorr()
        # err6
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold_B)


    def test_threshold_wrong_value(self):
        data_G = np.zeros((3,3,3))
        E_G = np.zeros((2,2))
        threshold5_B = -33.4
        threshold6_B = 33.4
        threshold7_B = [0.5,2]

        cls = SAM()
        # err7
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold5_B)
        # err7
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold6_B)
        # err8
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold7_B)

        cls = SID()
        # err7
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold5_B)
        # err7
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold6_B)
        # err8
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold7_B)

        cls = NormXCorr()
        # err7
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold5_B)
        # err7
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold6_B)
        # err8
        with self.assertRaises(ValueError):
            cls.classify(data_G, E_G, threshold=threshold7_B)


if __name__ == '__main__':
    unittest.main()
