#
#------------------------------------------------------------------------------
# Copyright (c) 2013-2014, Christian Therien
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#------------------------------------------------------------------------------
#
# utests.py - This file is part of the PySptools package.
#

import unittest
import numpy as np
from pysptools.eea import PPI, NFINDR, ATGP, FIPPI
from types import *

# B suffix == bad
# G suffix == good


class TestEeaError(unittest.TestCase):

    def setUp(self):
        pass

    def runTest(self):
        print('==> runTest: TestEeaError')
        self.test_transform()
        self.test_not_ndarray()
        self.test_wrong_input_dim()
        self.test_normalize_wrong_type()
        self.test_mask()
        self.test_plot()


    def test_transform(self):
        data = np.zeros((2,2,2))
        q = 2
        transform1 = np.zeros((2,2,2,2))
        transform2 = np.zeros((2,2,2))
        eea = NFINDR()
        # err16
        with self.assertRaises(RuntimeError):
            eea.extract(data, q, transform='string')
        # err17
        with self.assertRaises(RuntimeError):
            eea.extract(data, q, transform=transform1)
        # err18
        with self.assertRaises(RuntimeError):
            eea.extract(data, q, transform=transform2)


    def test_not_ndarray(self):
        data_G = np.zeros((2,2,2,2))
        data_B = [1,2]
        p_G = np.zeros((2,2))
        p_B = 12

        eea = PPI()
        # err1
        with self.assertRaises(RuntimeError):
            eea.extract(data_B, p_G)
        # err2
        with self.assertRaises(RuntimeError):
            eea.extract(data_G, p_B)

        eea = NFINDR()
        # err1
        with self.assertRaises(RuntimeError):
            eea.extract(data_B, p_G)
        # err2
        with self.assertRaises(RuntimeError):
            eea.extract(data_G, p_B)

        eea = ATGP()
        # err1
        with self.assertRaises(RuntimeError):
            eea.extract(data_B, p_G)
        # err2
        with self.assertRaises(RuntimeError):
            eea.extract(data_G, p_B)

        eea = FIPPI()
        # err1
        with self.assertRaises(RuntimeError):
            eea.extract(data_B, p_G)
        # err2
        with self.assertRaises(RuntimeError):
            eea.extract(data_G, p_B)


    def test_wrong_input_dim(self):
        data_B = np.zeros((2,2,2,2))
        p_G = 12

        eea = PPI()
        # err3
        with self.assertRaises(RuntimeError):
            eea.extract(data_B, p_G)

        eea = NFINDR()
        # err3
        with self.assertRaises(RuntimeError):
            eea.extract(data_B, p_G)

        eea = ATGP()
        # err3
        with self.assertRaises(RuntimeError):
            eea.extract(data_B, p_G)

        eea = FIPPI()
        # err3
        with self.assertRaises(RuntimeError):
            eea.extract(data_B, p_G)


    def test_normalize_wrong_type(self):
        data_G = np.zeros((3,3,3))
        p_G = 12
        normalize2_B = [1,2]

        eea = PPI()
        # err5
        with self.assertRaises(TypeError):
            eea.extract(data_G, p_G, normalize=normalize2_B)

        eea = NFINDR()
        # err5
        with self.assertRaises(TypeError):
            eea.extract(data_G, p_G, normalize=normalize2_B)

        eea = ATGP()
        # err5
        with self.assertRaises(TypeError):
            eea.extract(data_G, p_G, normalize=normalize2_B)

        eea = FIPPI()
        # err5
        with self.assertRaises(TypeError):
            eea.extract(data_G, p_G, normalize=normalize2_B)


    def test_mask(self):
        data_G = np.zeros((3,3,3))
        p_G = 12
        mask2_B = 123
        mask1_B = np.zeros((3,3,3))

        eea = NFINDR()
        # err13
        with self.assertRaises(RuntimeError):
            eea.extract(data_G, p_G, mask=mask2_B)
        # err14
        with self.assertRaises(RuntimeError):
            eea.extract(data_G, p_G, mask=mask1_B)

        eea = ATGP()
        # err13
        with self.assertRaises(RuntimeError):
            eea.extract(data_G, p_G, mask=mask2_B)
        # err14
        with self.assertRaises(RuntimeError):
            eea.extract(data_G, p_G, mask=mask1_B)


    def test_plot(self):
        info = {'wavelength':[1,2],'wavelength units':'cm'}

        eea = PPI()
        # err10
        with self.assertRaises(RuntimeError):
            eea.plot('../results', info)
        # err11
        eea.E = np.array([1,2])
        with self.assertRaises(TypeError):
            eea.plot('../results', info, suffix=[1,2])

        eea = NFINDR()
        # err10
        with self.assertRaises(RuntimeError):
            eea.plot('../results', info)
        # err11
        eea.E = np.array([1,2])
        with self.assertRaises(TypeError):
            eea.plot('../results', info, suffix=[1,2])

        eea = ATGP()
        # err10
        with self.assertRaises(RuntimeError):
            eea.plot('../results', info)
        # err11
        eea.E = np.array([1,2])
        with self.assertRaises(TypeError):
            eea.plot('../results', info, suffix=[1,2])

        eea = FIPPI()
        # err10
        with self.assertRaises(RuntimeError):
            eea.plot('../results', info)
        # err11
        eea.E = np.array([1,2])
        with self.assertRaises(TypeError):
            eea.plot('../results', info, suffix=[1,2])


if __name__ == '__main__':
    unittest.main()
