#
#------------------------------------------------------------------------------
# Copyright (c) 2013-2014, Christian Therien
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#------------------------------------------------------------------------------
#
# test_dnoise.py - This file is part of the PySptools package.
#

"""
The following functions are tested:
    SavitzkyGolay
    Whiten
    MNF
"""



import numpy as np
import os.path as osp
import pysptools.util as util
import pysptools.noise as ns
import pysptools.eea as eea
import pysptools.classification as cls


def SavitzkyGolay_spectra_filter(data, path):
    sg = ns.SavitzkyGolay()
#    fdata = sg.denoise_spectra(data, 11, 4, deriv=2)
    fdata = sg.denoise_spectra(data, 11, 4, deriv=0)
    sg.plot_spectrum_sample(data, path, 0, 1)
    return fdata


def SavitzkyGolay_bands_filter(data, path):
    sg = ns.SavitzkyGolay()
    fdata = sg.denoise_bands(data, 5, 2)
    sg.plot_bands_sample(path, 5)
    return fdata


def whiten(data):
    w = ns.Whiten()
    return w.apply(data)


def MNF(data, n_components, path):
    mnf = ns.MNF()
    mnf.apply(data)
    mnf.plot_components(path, 1)
    # get the first n_components
    return mnf.get_components(n_components)


def MNF_reduce_component_2_noise_and_invert(data):
    # Reduce the second component noise and
    # return the inverse transform
    mnf = ns.MNF()
    tdata = mnf.apply(data)
    dn = ns.SavitzkyGolay()
    tdata[:,:,1:2] = dn.denoise_bands(tdata[:,:,1:2], 15, 2)
    # inverse_transform remove the PCA rotation,
    # we obtain a whitened cube with
    # a noise reduction for the second component
    return mnf.inverse_transform(tdata)


def test_whiten(n_clusters, data, result_path, info):
    print('Testing whiten')
    wdata = whiten(data)
    km = cls.KMeans()
    km.predict(wdata, n_clusters)
    km.plot(result_path, colorMap='jet', suffix='whiten')


def test_MNF(n_clusters, n_components, data, result_path, info):
    print('Testing MNF')
    tdata = MNF(data, n_components, result_path)
    km = cls.KMeans()
    km.predict(tdata, n_clusters)
    km.plot(result_path, colorMap='jet', suffix='MNF')

    print('Testing MNF with component 2 noise reduction')
    idata = MNF_reduce_component_2_noise_and_invert(data)
    km = cls.KMeans()
    km.predict(idata, n_clusters)
    km.plot(result_path, colorMap='jet', suffix='MNF_with_component_2_noise_reduction')


def test_SavitzkyGolay(n_clusters, data, result_path, info):
    print('Testing SavitzkyGolay bands filter')

    tdata = SavitzkyGolay_bands_filter(data, result_path)
    km = cls.KMeans()
    km.predict(tdata, n_clusters)
    km.plot(result_path, colorMap='jet', suffix='SG_bands_filter')

    tdata = SavitzkyGolay_spectra_filter(data, result_path)
    km = cls.KMeans()
    km.predict(tdata, n_clusters)
    km.plot(result_path, colorMap='jet', suffix='SG_spectra_filter')


def tests_P2_7():
    data_path = r'..\data1'
    project_path = '..\\'
    result_path = osp.join(project_path, 'results')
    sample = 'SAMSON_part.hdr'
    #sample = '92AV3C.hdr'

    data_file = osp.join(data_path, sample)
    data, info = util.load_ENVI_file(data_file)

    n_clusters = 5
    km = cls.KMeans()
    km.predict(data, n_clusters)
    km.plot(result_path, colorMap='jet', suffix='data')

    n_components = 40
    test_MNF(n_clusters, n_components, data, result_path, info)
    test_whiten(n_clusters, data, result_path, info)
    test_SavitzkyGolay(n_clusters, data, result_path, info)


def tests_P3_3():
    import json
    data_path = r'..\data1'
    project_path = '..\\'
    result_path = osp.join(project_path, 'results')
    sample = 'SAMSON_part'

    # load the cube
    data_file = osp.join(data_path, sample+'.jdata')
    with open(data_file, 'r') as content_file:
        data = np.array(json.loads(content_file.read()))
    info_file = osp.join(data_path, sample+'.jhead')
    with open(info_file, 'r') as content_file:
        info = json.loads(content_file.read())

    n_clusters = 5
    km = cls.KMeans()
    km.predict(data, n_clusters)
    km.plot(result_path, colorMap='jet', suffix='data')

    n_components = 40
    test_MNF(n_clusters, n_components, data, result_path, info)
    test_whiten(n_clusters, data, result_path, info)
    test_SavitzkyGolay(n_clusters, data, result_path, info)


def tests():
    import sys
    if sys.version_info[:2] == (2,7):
        tests_P2_7()
    if sys.version_info[:2] == (3,3):
        tests_P3_3()


if __name__ == '__main__':
    import sys
    print(sys.version_info)
    if sys.version_info[:2] == (2,7):
        tests_P2_7()
    if sys.version_info[:2] == (3,3):
        tests_P3_3()
