#
#------------------------------------------------------------------------------
# Copyright (c) 2013-2014, Christian Therien
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#------------------------------------------------------------------------------
#
# utests.py - This file is part of the PySptools package.
#

import unittest
import numpy as np
from pysptools.detection import MatchedFilter, ACE, CEM, GLRT, OSP
from types import *

# B suffix == bad
# G suffix == good

class TestDetectionError(unittest.TestCase):

    def setUp(self):
        pass


    def runTest(self):
        print '==> runTest: TestDetectionError'
        self.test_not_ndarray()
        self.test_wrong_input_dim()
        self.test_threshold_wrong_type()
        self.test_plot()
        self.test_threshold_wrong_value()


    def test_not_ndarray(self):
        data_G = np.zeros((2,2,2,2))
        data_B = [1,2]
        t_G = np.zeros((3))
        t_B = 123
        U = np.zeros((3,3))

        dt = MatchedFilter()
        # err1
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, t_G)
        # err2
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, t_B)

        dt = ACE()
        # err1
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, t_G)
        # err2
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, t_B)

        dt = CEM()
        # err1
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, t_G)
        # err2
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, t_B)

        dt = GLRT()
        # err1
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, t_G)
        # err2
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, t_B)

        dt = OSP()
        # err1
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, U, t_G)
        # err2
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, U, t_B)


    def test_wrong_input_dim(self):
        data_B = np.zeros((2,2,2,2))
        data_G = np.zeros((3,3,3))
        t_G = np.zeros((3))
        t_B = np.zeros((1,2,1,2))
        U = np.zeros((3,3))

        dt = MatchedFilter()
        # err3
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, t_G)
        # err4
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, t_B)

        dt = ACE()
        # err3
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, t_G)
        # err4
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, t_B)

        dt = CEM()
        # err3
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, t_G)
        # err4
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, t_B)

        dt = GLRT()
        # err3
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, t_G)
        # err4
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, t_B)

        dt = OSP()
        # err3
        with self.assertRaises(RuntimeError):
            dt.detect(data_B, U, t_G)
        # err4
        with self.assertRaises(RuntimeError):
            dt.detect(data_G, U, t_B)


    def test_threshold_wrong_type(self):
        data_G = np.zeros((3,3,3))
        t_G = np.zeros((3))
        threshold_B = [1,2]
        U = np.zeros((3,3))

        dt = MatchedFilter()
        # err5
        with self.assertRaises(TypeError):
            dt.detect(data_G, t_G, threshold=threshold_B)

        dt = ACE()
        # err5
        with self.assertRaises(TypeError):
            dt.detect(data_G, t_G, threshold=threshold_B)

        dt = CEM()
        # err5
        with self.assertRaises(TypeError):
            dt.detect(data_G, t_G, threshold=threshold_B)

        dt = GLRT()
        # err5
        with self.assertRaises(TypeError):
            dt.detect(data_G, t_G, threshold=threshold_B)

        dt = OSP()
        # err5
        with self.assertRaises(TypeError):
            dt.detect(data_G, U, t_G, threshold=threshold_B)


    def test_plot(self):
        dt = MatchedFilter()
        # err10
        with self.assertRaises(RuntimeError):
            dt.plot('../results')
        # err11
        dt.target_map = np.array([1,2])
        with self.assertRaises(TypeError):
            dt.plot('../results', suffix=[1,2])

        dt = ACE()
        # err10
        with self.assertRaises(RuntimeError):
            dt.plot('../results')
        # err11
        dt.target_map = np.array([1,2])
        with self.assertRaises(TypeError):
            dt.plot('../results', suffix=[1,2])

        dt = CEM()
        # err10
        with self.assertRaises(RuntimeError):
            dt.plot('../results')
        # err11
        dt.target_map = np.array([1,2])
        with self.assertRaises(TypeError):
            dt.plot('../results', suffix=[1,2])

        dt = GLRT()
        # err10
        with self.assertRaises(RuntimeError):
            dt.plot('../results')
        # err11
        dt.target_map = np.array([1,2])
        with self.assertRaises(TypeError):
            dt.plot('../results', suffix=[1,2])

        dt = OSP()
        # err10
        with self.assertRaises(RuntimeError):
            dt.plot('../results')
        # err11
        dt.target_map = np.array([1,2])
        with self.assertRaises(TypeError):
            dt.plot('../results', suffix=[1,2])


    def test_threshold_wrong_value(self):
        data_G = np.zeros((3,3,3))
        t_G = np.zeros((3))
        threshold5_B = -33.4
        threshold6_B = 33.4
        U = np.zeros((3,3))

        dt = MatchedFilter()
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, t_G, threshold=threshold5_B)
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, t_G, threshold=threshold6_B)

        dt = ACE()
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, t_G, threshold=threshold5_B)
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, t_G, threshold=threshold6_B)

        dt = CEM()
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, t_G, threshold=threshold5_B)
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, t_G, threshold=threshold6_B)

        dt = GLRT()
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, t_G, threshold=threshold5_B)
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, t_G, threshold=threshold6_B)

        dt = OSP()
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, U, t_G, threshold=threshold5_B)
        # err7
        with self.assertRaises(ValueError):
            dt.detect(data_G, U, t_G, threshold=threshold6_B)


if __name__ == '__main__':
    unittest.main()

