/*
 * Decompiled with CFR 0.152.
 */
package stallone.datasequence.io;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.util.BitSet;
import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;
import stallone.api.datasequence.DataSequence;
import stallone.api.datasequence.IDataList;
import stallone.api.datasequence.IDataReader;
import stallone.api.datasequence.IDataSequence;
import stallone.api.doubles.Doubles;
import stallone.api.doubles.IDoubleArray;
import stallone.datasequence.io.DataReaderIterator;
import stallone.datasequence.io.DataReaderPairIterator;
import stallone.ints.PrimitiveIntTools;
import stallone.io.NicelyCachedRandomAccessFile;

public class DcdReader
implements IDataReader {
    private String filename;
    private boolean format__dcd_is_charmm;
    private boolean format__dcd_has_4dims;
    private boolean format__dcd_has_extra_block;
    private ByteOrder byteOrder;
    private NicelyCachedRandomAccessFile niceRandomAccessFile;
    private int numberOfFrames;
    private int startingTimestep;
    private int numberOfTimestepsBetweenSaves;
    private int numberOfAtoms;
    private int numberOfFixedAtoms;
    private double delta;
    private long frameAreaStartingPosition;
    private long framesizeRegular;
    private long framesizeWithFixed;
    private int nextFrameIndex;
    private int[] selection;
    private BitSet selected;
    private Logger logger;

    public DcdReader() {
    }

    public DcdReader(String _filename) throws IOException {
        this.filename = _filename;
        this.logger = Logger.getLogger(DcdReader.class.getName());
        this.initialize();
    }

    @Override
    public void setSource(String name) {
        this.filename = name;
    }

    @Override
    public void scan() throws IOException {
        this.initialize();
    }

    private void initialize() throws IOException {
        this.niceRandomAccessFile = new NicelyCachedRandomAccessFile(this.filename);
        this.byteOrder = this.detectEndianess();
        this.readHeader();
        this.niceRandomAccessFile.changePageSize((int)this.framesizeRegular);
        this.selection = PrimitiveIntTools.range(this.numberOfAtoms);
        this.selected = new BitSet(this.numberOfAtoms);
        this.selected.set(0, this.numberOfAtoms);
    }

    private ByteOrder detectEndianess() throws IOException {
        this.niceRandomAccessFile.seek(0L);
        ByteBuffer recordLengthBuffer = this.niceRandomAccessFile.readToBuffer(4);
        this.niceRandomAccessFile.seek(0L);
        if (recordLengthBuffer.order(ByteOrder.LITTLE_ENDIAN).getInt() == 84) {
            this.logger.log(Level.FINE, "Little endian encoding.");
            return ByteOrder.LITTLE_ENDIAN;
        }
        if (recordLengthBuffer.order(ByteOrder.BIG_ENDIAN).getInt() == 84) {
            this.logger.log(Level.FINE, "Big endian encoding.");
            return ByteOrder.BIG_ENDIAN;
        }
        throw new IOException("Invalid record length of first record.");
    }

    private ByteBuffer readNextRecord() throws IOException {
        int recordLength = this.niceRandomAccessFile.readToBuffer(4).order(this.byteOrder).getInt();
        ByteBuffer recordByteBuffer = this.niceRandomAccessFile.readToBuffer(recordLength).order(this.byteOrder);
        int verifyRecordLength = this.niceRandomAccessFile.readToBuffer(4).order(this.byteOrder).getInt();
        if (recordLength == verifyRecordLength) {
            this.logger.log(Level.FINEST, "Record (length=" + recordLength + ") found and read.");
            return recordByteBuffer;
        }
        throw new IOException("Invalid file data.");
    }

    private void skipNextRecord() throws IOException {
        int recordLength = this.niceRandomAccessFile.readToBuffer(4).order(this.byteOrder).getInt();
        this.niceRandomAccessFile.skipBytes(recordLength);
        int verifyRecordLength = this.niceRandomAccessFile.readToBuffer(4).order(this.byteOrder).getInt();
        if (recordLength == verifyRecordLength) {
            this.logger.log(Level.FINEST, "Record (length=" + recordLength + ") found and skipped.");
            return;
        }
        throw new IOException("Invalid file data.");
    }

    private void readHeader() throws IOException {
        this.format__dcd_is_charmm = false;
        this.format__dcd_has_4dims = false;
        this.format__dcd_has_extra_block = false;
        ByteBuffer mainHeaderRecord = this.readNextRecord();
        if (mainHeaderRecord.remaining() == 84 && mainHeaderRecord.get(0) == 67 && mainHeaderRecord.get(1) == 79 && mainHeaderRecord.get(2) == 82 && mainHeaderRecord.get(3) == 68) {
            if (mainHeaderRecord.getInt(80) != 0) {
                this.format__dcd_is_charmm = true;
                this.logger.log(Level.FINE, "CHARMM format DCD file (also NAMD 2.1 and later).");
                if (mainHeaderRecord.getInt(44) != 0) {
                    this.format__dcd_has_extra_block = true;
                    this.logger.log(Level.FINE, "  Has extra block.");
                }
                if (mainHeaderRecord.getInt(48) != 0) {
                    this.format__dcd_has_4dims = true;
                    this.logger.log(Level.FINE, "  Has 4-dims.");
                }
            } else {
                this.logger.log(Level.FINE, "X-PLOR format DCD file (also NAMD 2.0 and earlier).");
            }
            this.numberOfFrames = mainHeaderRecord.getInt(4);
            this.logger.log(Level.FINE, "  Number of sets of coordinates: " + this.numberOfFrames);
            this.startingTimestep = mainHeaderRecord.getInt(8);
            this.logger.log(Level.FINE, "  Starting timestep            : " + this.startingTimestep);
            this.numberOfTimestepsBetweenSaves = mainHeaderRecord.getInt(12);
            this.logger.log(Level.FINE, "  Timesteps between dcd saves  : " + this.numberOfTimestepsBetweenSaves);
            this.numberOfFixedAtoms = mainHeaderRecord.getInt(36);
            this.logger.log(Level.FINE, "  Number of fixed atoms        : " + this.numberOfFixedAtoms);
            if (this.numberOfFixedAtoms > 0) {
                throw new RuntimeException("Sorry, read of fixed atom coordinates currently not implemented.");
            }
            if (this.format__dcd_is_charmm) {
                float ftmp = mainHeaderRecord.getFloat(40);
                this.delta = ftmp;
            } else {
                this.delta = mainHeaderRecord.getDouble(40);
            }
            this.logger.log(Level.FINE, "  Timestep                     : " + this.delta);
            ByteBuffer descriptionHeaderRecord = this.readNextRecord();
            int lines = descriptionHeaderRecord.getInt();
            byte[] line = new byte[80];
            int i = 0;
            while (i < lines) {
                descriptionHeaderRecord.get(line);
                ++i;
            }
            ByteBuffer noOfAtomsRecord = this.readNextRecord();
            this.numberOfAtoms = noOfAtomsRecord.getInt(0);
            this.logger.log(Level.FINE, "No of atoms : " + this.numberOfAtoms);
            this.frameAreaStartingPosition = this.niceRandomAccessFile.getFilePointer();
            this.framesizeRegular = 0L;
            long dims = 3L;
            if (this.format__dcd_has_4dims) {
                dims = 4L;
            }
            long fullExtraframesize = 0L;
            if (this.format__dcd_has_extra_block) {
                fullExtraframesize = 56L;
            }
            long cordRecordSize = (this.numberOfAtoms - this.numberOfFixedAtoms) * 4;
            long fullRecordSize = 4L + cordRecordSize + 4L;
            long cordRecordSizeWithFixed = this.numberOfAtoms * 4;
            long fullRecordSizeWithFixed = 4L + cordRecordSizeWithFixed + 4L;
            this.framesizeWithFixed = fullRecordSizeWithFixed * dims + fullExtraframesize;
            this.framesizeRegular = fullRecordSize * dims + fullExtraframesize;
            this.nextFrameIndex = 0;
            long endpos = this.frameAreaStartingPosition + (long)this.numberOfFrames * this.framesizeRegular;
            if (this.niceRandomAccessFile.length() != endpos) {
                throw new IOException("Real file size does not match the calculated file size. Maybe an interrupt has occured during creation of trajectory file.Real: " + this.niceRandomAccessFile.length() + "  Expected: " + endpos);
            }
        } else {
            throw new IOException("Invalid file format. Header corrupted or not found.");
        }
        this.logger.log(Level.FINE, "Checking file size vs. calculated number of frames .... OK.");
    }

    private IDoubleArray readNextFrame(IDoubleArray target) throws IOException {
        this.niceRandomAccessFile.fitsInBuffer((int)this.framesizeRegular);
        if (this.format__dcd_has_extra_block) {
            this.skipNextRecord();
        }
        int n = this.numberOfAtoms - this.numberOfFixedAtoms;
        FloatBuffer xBuffer = this.readNextRecord().asFloatBuffer();
        FloatBuffer yBuffer = this.readNextRecord().asFloatBuffer();
        FloatBuffer zBuffer = this.readNextRecord().asFloatBuffer();
        int r = 0;
        int i = 0;
        while (i < n) {
            float x = xBuffer.get();
            float y = yBuffer.get();
            float z = zBuffer.get();
            if (this.selected.get(i)) {
                target.set(r, 0, x);
                target.set(r, 1, y);
                target.set(r, 2, z);
                ++r;
            }
            ++i;
        }
        ++this.nextFrameIndex;
        if (this.format__dcd_has_4dims) {
            this.skipNextRecord();
        }
        return target;
    }

    private long getPositionOfFrame(int frameIndex) {
        if (frameIndex > 0) {
            return this.frameAreaStartingPosition + this.framesizeWithFixed + (long)(frameIndex - 1) * this.framesizeRegular;
        }
        if (frameIndex == 0) {
            return this.frameAreaStartingPosition;
        }
        throw new IndexOutOfBoundsException("Index of frame is negative.");
    }

    @Override
    public int size() {
        return this.numberOfFrames;
    }

    @Override
    public int dimension() {
        return this.numberOfAtoms * 3;
    }

    public double getTimeOfFrame(int frameIndex) {
        int currentStep = this.startingTimestep + this.numberOfTimestepsBetweenSaves * frameIndex;
        return (double)currentStep * this.delta;
    }

    public IDoubleArray get(int frameIndex, IDoubleArray target) {
        try {
            if (frameIndex < 0 || this.numberOfFrames <= frameIndex) {
                throw new IndexOutOfBoundsException("Invalid frame index " + frameIndex + ".");
            }
            if (frameIndex != this.nextFrameIndex) {
                this.nextFrameIndex = frameIndex;
                this.niceRandomAccessFile.seek(this.getPositionOfFrame(frameIndex));
                this.logger.log(Level.FINEST, "Repositioning for frame " + frameIndex + " to " + this.niceRandomAccessFile.getFilePointer());
            } else {
                this.logger.log(Level.FINEST, "File position is okay. Frame " + frameIndex + " is at " + this.niceRandomAccessFile.getFilePointer());
            }
            return this.readNextFrame(target);
        }
        catch (IOException ex) {
            this.logger.log(Level.SEVERE, null, ex);
            throw new RuntimeException("Problems reading DCD, caught I/O exception. Message: " + ex.getMessage());
        }
    }

    @Override
    public void close() throws IOException {
        this.niceRandomAccessFile.close();
    }

    @Override
    public void open() throws IOException {
        this.niceRandomAccessFile.open();
    }

    @Override
    public double getTime(int frameIndex) {
        return this.delta * (double)frameIndex;
    }

    @Override
    public void select(int[] _selection) {
        if (_selection == null) {
            _selection = PrimitiveIntTools.range(this.numberOfAtoms);
        }
        this.selection = _selection;
        this.selected.clear();
        int i = 0;
        while (i < _selection.length) {
            this.selected.set(_selection[i]);
            ++i;
        }
    }

    @Override
    public int[] getSelection() {
        return this.selection;
    }

    @Override
    public IDoubleArray get(int frameIndex) {
        return this.get(frameIndex, Doubles.create.array(this.selection.length, 3));
    }

    @Override
    public IDoubleArray getView(int index) {
        return this.get(index);
    }

    @Override
    public Iterator<IDoubleArray> iterator() {
        return new DataReaderIterator(this);
    }

    @Override
    public Iterator<IDoubleArray[]> pairIterator(int spacing) {
        return new DataReaderPairIterator(this, spacing);
    }

    @Override
    public Iterable<IDoubleArray[]> pairs(int spacing) {
        class PairIterable
        implements Iterable<IDoubleArray[]> {
            private IDataReader seq;
            private int spacing = 1;

            public PairIterable(IDataReader _seq, int _spacing) {
                this.seq = _seq;
                this.spacing = _spacing;
            }

            @Override
            public Iterator<IDoubleArray[]> iterator() {
                return new DataReaderPairIterator(this.seq, this.spacing);
            }
        }
        return new PairIterable(this, spacing);
    }

    @Override
    public IDataSequence load() {
        IDataList res = DataSequence.create.list();
        int i = 0;
        while (i < this.size()) {
            res.add(this.get(i));
            ++i;
        }
        return res;
    }

    public String getFileName() {
        return this.filename;
    }

    @Override
    public long memorySize() {
        return this.size() * this.dimension() * 8;
    }
}

