/** @file swephelp.h
** @brief swephelp export header file.
**
** Main swephelp header, imports all swephelp functions.
**
** @author Stanislas Marquis <stnsls@gmail.com>
** @date 19.11.2008
** @remarks Based on swisseph version 1.75.00
*/

#ifndef _SWEPHELP_H
#define _SWEPHELP_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <swephexp.h>

/* aspects */
#define SWH_ASP_CONJUNCTION 0
#define SWH_ASP_SQUISEXTILE 15
#define SWH_ASP_SEMINOVILE 20
#define SWH_ASP_SQUISQUARE 22.5
#define SWH_ASP_UNDECILE 360/11.0
#define SWH_ASP_SEMISEXTILE 30
#define SWH_ASP_SEMIQUINTILE 36
#define SWH_ASP_NOVILE 40
#define SWH_ASP_SEMISQUARE 45
#define SWH_ASP_SEPTILE 360/7.0
#define SWH_ASP_SEXTILE 60
#define SWH_ASP_BIUNDECILE ((360/11.0)*2)
#define SWH_ASP_QUINTILE 72
#define SWH_ASP_BINOVILE 80
#define SWH_ASP_SQUARE 90
#define SWH_ASP_TRIUNDECILE ((360/11.0)*3)
#define SWH_ASP_BISEPTILE ((360/7.0)*2)
#define SWH_ASP_TRINE 120
#define SWH_ASP_QUADUNDECILE ((360/11.0)*4)
#define SWH_ASP_SESQUISQUARE 135
#define SWH_ASP_BIQUINTILE 144
#define SWH_ASP_QUINCUNX 150
#define SWH_ASP_TRISEPTILE ((360/7.0)*3)
#define SWH_ASP_QUATRONOVILE 160
#define SWH_ASP_QUINUNDECILE ((360/11.0)*5)
#define SWH_ASP_OPPOSITION 180

/* from swhdatetime.c */

/** @brief Get current Julian day number, Gregorian calendar.
** @return Julian day number
*/
double swh_utcnow(void);

/** @brief Get Julian day number from a date and time.
**
** A conveniant function that, compared to swe_julday, does not require
** hour in decimal format, but accepts integers only.
**
** @attention Does not check date validity. See swe_date_conversion.
**
** @see swh_revjul()
**
** @param year Year (4 digits)
** @param mon Month [1;12]
** @param day Day [1;31]
** @param hour Hour [0;23]
** @param min Minutes [0;59]
** @param sec Seconds [0;59]
** @param flag Calendar type (SE_GREG_CAL|SE_JUL_CAL)
** @return Julian day number
*/
double swh_julday(int year, int mon, int day, int hour, int min,
	int sec, int flag);

/** @brief Reverse Julian day to date and time.
**
** Similar to swe_revjul, but returns time with three integers instead
** of one double.
**
** @see swh_julday()
**
** @param jd Julian day
** @param flag Calendar type (SE_GREG_CAL|SE_JUL_CAL)
** @param dt Results, declared as int[6] (year, month, day, hour, min, sec)
** @return 0
*/
int swh_revjul(double jd, int flag, int *dt);

/* from swhfmt.c */

/** @brief Get degrees, sign number, minutes, seconds, from longitude position.
**
** Split a longitude position to a more usual format.
**
** @param pos Longitude position [0;360[
** @param ret Results, declared as int[4] (deg, sign, min, sec)
** @return 0
*/
int swh_degsplit(double pos, int *ret);

/** @brief Get the 3-letters string representing a sign number.
**
** @param sign Sign number [0;11]
** @param str Returned string, declared as char[4]
** @return 0, or -1 if sign was invalid
*/
int swh_signtostr(int sign, char *str);

/** @brief Get house system name.
**
** @param hsys House system char [PKRCBOAEHVXGU]
** @param str Returned string, declared as char[25]
** @return 0, or -1 if house system is unknown
*/
int swh_get_house_system_name(char hsys, char *str);

/* from swhsearch.c */

/** @brief Get the minimum retrogradation time of a planet, in days.
**
** This function is mainly used in swephelp search routines.
**
** The returned value is ~3 days inferior to the "real" minimum
** retrogradation time (tested from 1.1.-3000 to 1.1.3000 when possible).
**
** @remarks May return 10 (default) if the planet is not implemented.
** This may be the case if you want some unusual asteroid. If you need to
** speed search routines, you may try to compute the retrogradation time
** using the minretro.py script, add one line to the source code, and recompile.
**
** @todo Implement more planets, main asteroids...
**
** @param planet Planet number (SE_*, etc)
** @param err Buffer for errors, declared as char[32] (or more)
** @return Number of days, or -1 if planet is never retrograde.
*/
int swh_min_retro_time(int planet, char *err);

/** @brief Get the maximum retrogradation time of a planet, in days.
**
** Thought this function would be usefull in search routines, but
** seems like it is not (?).
**
** @remarks The returned value is ~2 days superior to the "real" maximum
** retrogradation time (tested from 1.1.-3000 to 1.1.3000 when possible).
**
** @deprecated This function is not usefull. May disappear one day.
**
** @param planet Planet number (SE_*, etc)
** @param err Buffer for errors, declared as char[32] (or more)
** @return Number of days, or -1 if planet is never retrograde.
*/
int swh_max_retro_time(int planet, char *err);

/** @brief Find next direction changing of object.
**
** This function tries to find when and where a planet in direct or
** retrograde motion goes retrograde or direct (respectively).
**
** @attention If step is set to 0, use a predefined step (recommended); it must
** be inferior to the planet's minimum retrogradation time.
**
** @remarks If dayspan is set to 0, the search is not limited in time.
** Otherwise, the function may return 1 when time limit has been reached.
** Flag must include SEFLG_SPEED, and SEFLG_NOGDEFL to avoid bad surprises;
** alternatively use true positions.
**
** @see swh_min_retro_time()
**
** @param planet Planet number (SE_*, etc)
** @param jdstart Julian day number, when search is starting
** @param step Number of days used in the dichotomic search process
** @param backw Search before jdstart [1], or after [0] (boolean)
** @param dayspan Limit search to a certain time, expressed in days
** @param flag Calculation flags, see swisseph docs
** @param jdret Julian day number found
** @param posret Planet's positions found
** @param err Buffer for error, declared as char[256]
** @return 0 on success, 1 if time limit reached, -1 on error
*/
int swh_next_retro(int planet, double jdstart, double step, int backw,
	double dayspan, int flag, double *jdret, double *posret, char *err);

/** @brief Get an estimation of when a planet is gone past a fixed point.
**
** Get Julian day number and positions when a celestial object is gone past
** a fixed point expressed in longitude degrees. This is not meant to be
** precise at all. Mainly used in swephelp search routines.
**
** @attention Same warning as in swh_next_retro.
**
** @remarks If step is set to 0, use a predefined step (recommended).
**
** @param planet Planet number (SE_*, etc)
** @param fixedpt Fixed pointed targeted [0;360[
** @param jdstart Julian day number, when search is starting
** @param step Number of days used in the dichotomic search process
** @param backw Search before jdstart [1], or after [0] (boolean)
** @param flag Calculation flags, see swisseph docs
** @param jdret Julian day number found
** @param posret Planet's positions found
** @param err Buffer for errors, declared as char[256]
** @return 0 on success, -1 on error
*/
int swh_go_past(int planet, double fixedpt, double jdstart, double step,
	int backw, int flag, double *jdret, double *posret, char *err);

/** @brief Find next exact aspect to a fixed point.
**
** Get Julian day number and positions when a celestial object makes a
** longitudinal aspect to a fixed point expressed in longitude degrees.
**
** @attention Same warning as in swh_next_retro.
**
** @remarks If step is set to 0, use a predefined step (recommended).
** If dayspan is set to 0, the search is not limited in time.
** Otherwise, the function may return 1 when time limit has been reached.
**
** @param planet Planet number (SE_*, etc)
** @param aspect Aspect, in degrees [0;360[
** @param fixedpt Fixed point targeted [0;360[
** @param jdstart Julian day number, when search is starting
** @param step Number of days used in the dichotomic search process
** @param backw Search before jdstart [1], or after [0] (boolean)
** @param dayspan Limit search to a certain time, expressed in days
** @param flag Calculation flags, see swisseph docs
** @param jdret Julian day number found
** @param posret Planet's positions found
** @param err Buffer for errors, declared as char[256]
** @return 0 on success, 1 if time limit reached, -1 on error
*/
int swh_next_aspect(int planet, double aspect, double fixedpt, double jdstart,
	double step, int backw, double dayspan, int flag, double *jdret,
	double *posret, char *err);

/** @brief Find next exact aspect to a fixed point.
**
** Same as swh_next_aspect but with aspect in [0;180], instead of [0;360[.
**
** @see swh_next_aspect()
**
** @remarks If aspect is not 0 or 180, it will try two aspects [0;360[,
** and return the nearest from jdstart. It may then be faster to use
** swh_next_aspect several times, especially when scanning long periods of time.
*/
int swh_next_aspect2(int planet, double aspect, double fixedpt, double jdstart,
	double step, int backw, double dayspan, int flag, double *jdret,
	double *posret, char *err);

/** @brief Find next aspect between two moving objects.
**
** Get Julian day number and positions when a celestial object makes a
** longitudinal aspect to another moving object.
**
** @attention Here, step may not be set to 0. If you estimate that the aspect
** is to occur in a very long time, you better set it to a high value, for
** faster results. In doubt, set it to 10.
**
** @remarks If star != "", the other planet is ignored.
** If dayspan is set to 0, the search is not limited in time.
** Otherwise, the function may return 1 when time limit has been reached.
**
** @param planet Planet number (SE_*, etc)
** @param aspect Aspect, in degrees [0;360[
** @param other Other planet number
** @param star Fixed star
** @param jdstart Julian day number, when search is starting
** @param step Number of days used in the dichotomic search process
** @param backw Search before jdstart [1], or after [0] (boolean)
** @param dayspan Limit search to a certain time, expressed in days
** @param flag Calculation flags, see swisseph docs
** @param jdret Julian day number found
** @param posret0 Planet's positions found
** @param posret1 Other planet (or star) positions found
** @param err Buffer for errors, declared as char[256]
** @return 0 on success, 1 if time limit reached, -1 on error
*/
int swh_next_aspect_with(int planet, double aspect, int other, char *star,
	double jdstart, double step, int backw, double dayspan, int flag,
	double *jdret, double *posret0, double *posret1, char *err);

/** @brief Find next aspect between two moving objects.
**
** Same as swh_next_aspect_with, but aspect in [0;180], instead of [0;360[.
**
** @see swh_next_aspect_with()
**
** @remarks If aspect is not 0 or 180, it will try two aspects [0;360[, and
** return the nearest from jdstart. It may then be faster to use
** swh_next_aspect_with several times, especially when scanning
** long periods of time.
*/
int swh_next_aspect_with2(int planet, double aspect, int other, char *star,
	double jdstart, double step, int backw, double dayspan, int flag,
	double *jdret, double *posret0, double *posret1, char *err);

/** @brief Find next aspect to a house cusp.
**
** Get Julian day number and positions, and houses cusps, when a
** celestial object makes longitudinal aspect to a house cusp.
**
** @attention Here, step may not be too high, or 0. Setting it to 0.2 is enough.
**
** @remarks If star != "", the planet is ignored.
**
** @see For risings, settings, meridian transits, see swe_rise_trans.
**
** @param planet Planet number (SE_*, etc)
** @param star Fixed star
** @param aspect Aspect, in degrees [0;360[
** @param cusp House cusp number [1;12[, or [1;36[ for Gauquelin sectors
** @param jdstart Julian day number, when search is starting
** @param lat Latitude, in degrees (north is positive)
** @param lon Longitude, in degrees (east is positive)
** @param hsys House system, see swisseph docs
** @param step Number of days used in the dichotomic search process
** @param backw Search before jdstart [1], or after [0] (boolean)
** @param flag Calculation flags, see swisseph docs
** @param jdret Julian day number found
** @param posret Planet (or fixed star) positions found
** @param cuspsret House cusps positions found
** @param ascmcret Asc-Mc-etc found, see swisseph docs
** @param err Buffer for errors, declared as char[256]
** @return 0 on success, -1 on error
*/
int swh_next_aspect_cusp(int planet, char *star, double aspect, int cusp,
	double jdstart, double lat, double lon, int hsys, double step, int backw,
	int flag, double *jdret, double *posret, double *cuspsret, double *ascmcret,
	char *err);

/** @brief Find next aspect to a house cusp.
**
** Same as swh_next_aspect_cusp, but aspect in [0;180], instead of [0;360[.
**
** @see swh_next_aspect_cusp()
**
** @remarks If aspect is not 0 or 180, it will try two aspects [0;360[, and
** return the nearest from jdstart. It may then be faster to use
** swh_next_aspect_cusp several times, especially when scanning
** long periods of time.
*/
int swh_next_aspect_cusp2(int planet, char *star, double aspect, int cusp,
	double jdstart, double lat, double lon, int hsys, double step, int backw,
	int flag, double *jdret, double *posret, double *cuspsret, double *ascmcret,
	char *err);

/** @brief Aspect matching.
**
** Check if the two given positions match the aspect within the given orb.
** This also calculates the difference between targeted aspect and distance
** between objects, and if aspect is applicating or separating, or stable due
** to equal speeds.
**
** @param pos0 First object longitude, in degrees [0;360[
** @param speed0 First object longitude speed, in degrees per day
** @param pos1 Second object longitude, in degrees [0;360[
** @param speed1 Second object longitude speed, in degrees per day
** @param aspect Aspect targeted, in degrees [0;360[
** @param orb Orb allowed, in degrees
** @param diffret Difference between aspect and objects distance, in degrees
** @param applic Aspect applicating [-1], separating [1], or stable [0]
** @param factor Aspect strength, compared to orb
** @return 0 if aspect match, else -1
*/
int swh_match_aspect(double pos0, double speed0, double pos1, double speed1,
	double aspect, double orb, double *diffret, int *applic, double *factor);

/** @brief Aspect matching.
**
** Same as swh_match_aspect, but aspect in [0;180], instead of [0;360[
**
** @see swh_match_aspect()
*/
int swh_match_aspect2(double pos0, double speed0, double pos1, double speed1,
	double aspect, double orb, double *diffret, int *applic, double *factor);

/** @brief Get number of years difference between two julian days.
**
** One exact "astrological" year can be considered as one solar return.
** Then is it varying with the type of zodiac in use.
**
** @param jd1 First Julian day
** @param jd2 Second Julian day
** @param flag Calculation flag
** @param years Result, declared as double
** @param err Buffer for errors, declared as char[256]
** @return 0 on success, else -1
*/
int swh_years_diff(double jd1, double jd2, int flag, double *years, char *err);

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* swephelp.h */

/* END. */
