/** @file swhdatetime.c
** @brief swephelp date and time functions.
**
** @author Stanislas Marquis <stnsls@gmail.com>
** @date 19.11.2008
*/

#ifdef __cplusplus
extern "C"
{
#endif

#include <swephexp.h>
#include <time.h>

/** @brief Get current Julian day number, Gregorian calendar.
** @return Julian day number
*/
double swh_utcnow(void)
{
	time_t t;
	t = time(NULL);
#ifdef WIN32
	struct tm *tmp = gmtime(&t);
	return swe_julday(tmp->tm_year+1900, tmp->tm_mon+1, tmp->tm_day,
		(tmp->tm_hour+(tmp->tm_min/60.0)+(tmp->tm_sec/3600.0)), SE_GREG_CAL);
#else
	struct tm tmp;
	gmtime_r(&t, &tmp);
	return swe_julday(tmp.tm_year+1900, tmp.tm_mon+1, tmp.tm_mday,
		(tmp.tm_hour+(tmp.tm_min/60.0)+(tmp.tm_sec/3600.0)), SE_GREG_CAL);
#endif /* WIN32 */
}

/** @brief Get Julian day number from a date and time.
**
** A conveniant function that, compared to swe_julday, does not require
** hour in decimal format, but accepts integers only.
**
** @attention Does not check date validity. See swe_date_conversion.
**
** @see swh_revjul()
**
** @param year Year (4 digits)
** @param mon Month [1;12]
** @param day Day [1;31]
** @param hour Hour [0;23]
** @param min Minutes [0;59]
** @param sec Seconds [0;59]
** @param flag Calendar type (SE_GREG_CAL|SE_JUL_CAL)
** @return Julian day number
*/
double swh_julday(int year, int mon, int day, int hour, int min,
	int sec, int flag)
{
	return swe_julday(year, mon, day, (hour+(min/60.0)+(sec/3600.0)), flag);
}

/** @brief Reverse Julian day to date and time.
**
** Similar to swe_revjul, but returns time with three integers instead
** of one double.
**
** @see swh_julday()
**
** @param jd Julian day
** @param flag Calendar type (SE_GREG_CAL|SE_JUL_CAL)
** @param dt Results, declared as int[6] (year, month, day, hour, min, sec)
** @return 0
*/
int swh_revjul(double jd, int flag, int *dt)
{
	double t;
	swe_revjul(jd, flag, &dt[0], &dt[1], &dt[2], &t);
	dt[3] = (int) floor(t);
	t -= dt[3];
	dt[4] = (int) floor(t*60);
	t -= dt[4]/60.0;
	dt[5] = (int) floor(t*3600);
	return 0;
}

#ifdef __cplusplus
} /* extern "C" */
#endif

/* END */
