/** @file swhfmt.c
** @brief swephelp formatting functions.
**
** @author Stanislas Marquis <stnsls@gmail.com>
** @date 29.10.2008
*/

#ifdef __cplusplus
extern "C"
{
#endif

#include <swephexp.h>

/** @brief Get degrees, sign number, minutes, seconds, from longitude position.
**
** Split a longitude position to a more usual format.
**
** @param pos Longitude position [0;360[
** @param ret Results, declared as int[4] (deg, sign, min, sec)
** @return 0
*/
int swh_degsplit(double pos, int *ret)
{
	pos = swe_degnorm(pos);
	ret[1] = (int) floor(pos/30.0);
	pos -= ret[1]*30;
	ret[0] = (int) floor(pos);
	pos -= ret[0];
	ret[2] = (int) floor(pos*60);
	pos -= ret[2]/60.0;
	ret[3] = (int) floor(pos*3600);
	return 0;
}

/** @brief Get the 3-letters string representing a sign number.
**
** @param sign Sign number [0;11]
** @param str Returned string, declared as char[4]
** @return 0, or -1 if sign was invalid
*/
int swh_signtostr(int sign, char *str)
{
	switch (sign)
	{
		case 0: strcpy(str, "Ari"); break;
		case 1: strcpy(str, "Tau"); break;
		case 2: strcpy(str, "Gem"); break;
		case 3: strcpy(str, "Can"); break;
		case 4: strcpy(str, "Leo"); break;
		case 5: strcpy(str, "Vir"); break;
		case 6: strcpy(str, "Lib"); break;
		case 7: strcpy(str, "Sco"); break;
		case 8: strcpy(str, "Sag"); break;
		case 9: strcpy(str, "Cap"); break;
		case 10: strcpy(str, "Aqu"); break;
		case 11: strcpy(str, "Pis"); break;
		default: return -1;
	}
	return 0;
}

/** @brief Get house system name.
**
** @param hsys House system char [PKRCBOAEHVXGU]
** @param str Returned string, declared as char[25]
** @return 0, or -1 if house system is unknown
*/
int swh_get_house_system_name(char hsys, char *str)
{
	switch (hsys)
	{
		case 'P': strcpy(str, "Placidus"); break;
		case 'K': strcpy(str, "Koch"); break;
		case 'R': strcpy(str, "Regiomontanus"); break;
		case 'C': strcpy(str, "Campanus"); break;
		case 'B': strcpy(str, "Alcabitius"); break;
		case 'O': strcpy(str, "Porphyry"); break;
		case 'A': strcpy(str, "Equal"); break;
		case 'E': strcpy(str, "Equal"); break;
		case 'H': strcpy(str, "Horizon/azimuth"); break;
		case 'V': strcpy(str, "Equal Vehlow"); break;
		case 'X': strcpy(str, "Axial rotation/meridian"); break;
		case 'G': strcpy(str, "Gauquelin"); break;
		case 'U': strcpy(str, "Krusinski"); break;
		case 'W': strcpy(str, "Whole sign"); break;
		default: return -1;
	}
	return 0;
}

#ifdef __cplusplus
} /* extern "C" */
#endif

/* END. */
