/*
    Swephelp

    Copyright 2007-2009 Stanislas Marquis <stnsls@gmail.com>

    Swephelp is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    Swephelp is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Swephelp.  If not, see <http://www.gnu.org/licenses/>.
*/

/** @file swhdatetime.c
** @brief swephelp date and time functions
**
** @author Stanislas Marquis <stnsls@gmail.com>
** @date 11.01.2009
*/

#ifdef __cplusplus
extern "C"
{
#endif

#include "swhdatetime.h"
#include <swephexp.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

/** @brief Get current Julian day number, Gregorian calendar
** @return Julian day number
*/
double swh_jdnow(void)
{
	time_t t = time(NULL);
#ifdef WIN32 /* has not gmtime_r ? */
	struct tm *tmp = gmtime(&t);
	return swe_julday(tmp->tm_year+1900, tmp->tm_mon+1, tmp->tm_mday,
		(tmp->tm_hour+(tmp->tm_min/60.0)+(tmp->tm_sec/3600.0)), SE_GREG_CAL);
#else
	struct tm tmp;
	gmtime_r(&t, &tmp);
	return swe_julday(tmp.tm_year+1900, tmp.tm_mon+1, tmp.tm_mday,
		(tmp.tm_hour+(tmp.tm_min/60.0)+(tmp.tm_sec/3600.0)), SE_GREG_CAL);
#endif
}

/** @brief Reverse Julian day to date and time
**
** Similar to swe_revjul, but returns time with three integers instead
** of one double. (Also tries to avoid some floating points rounding errors.)
**
** @see swh_julday()
**
** @param jd Julian day
** @param flag Calendar type (SE_GREG_CAL|SE_JUL_CAL)
** @param dt Results, declared as int[6] (year, month, day, hour, min, sec)
** @return 0
*/
int swh_revjul(double jd, int flag, int *dt)
{
	assert(flag == SE_GREG_CAL || flag == SE_JUL_CAL);
	double t;
	swe_revjul(jd, flag, &dt[0], &dt[1], &dt[2], &t);
	dt[3] = (int) floor(t);
	t -= dt[3];
	dt[4] = (int) floor(t * 60);
	t -= dt[4]/60.0;
	dt[5] = (int) lround(t * 3600);
	if (dt[5] == 60) /* rounding error */
	{
		dt[5] = 0;
		dt[4] += 1;
		if (dt[4] == 60)
		{
			dt[4] = 0;
			dt[3] += 1;
			/* wont go further? */
		}
	}
	return 0;
}

/** @brief UtcDt constructor
**
** This will allocate memory for an UtcDt object and return it. Internal
** variables will be set to no timezone (empty strings), gregorian calendar,
** no offset and unknown daylight flag. User must fill in the tm struct with
** date and time information and possibly the timezone (given as zoneinfo file
** ex: ":Europe/Zurich" or any suitable for tzset) or the offset, eventually
** the isdst flag....
**
** @return new UtcDt
*/
swh_UtcDt *swh_newUtcDt(void)
{
	swh_UtcDt *ret = (swh_UtcDt*) malloc(sizeof(swh_UtcDt));
	strcpy(ret->tz, "");
	strcpy(ret->tzname, "");
	ret->cal = 1; /* gregorian */
	ret->offset = 0; /* no offset */
	ret->isdst = -1; /* unknown or not needed */
	ret->status = -1; /* not valid utc, is local time. Needs to be mk'ed. */
	return ret;
}

/** @brief Get Julian day from given UtcDt
**
** Calculate Julian day for given UtcDt object. By the way, out of bound values
** are normalized.
**
** @return 0 if success, -1 on error, -2 needs user-defined dst
*/
int swh_mkUtcDt(swh_UtcDt *utcdt, double *jdret)
{
	int i;
	if (utcdt->status == 0) /* is valid utc */
	{
		i = swe_date_conversion(utcdt->dt.tm_year+1900, utcdt->dt.tm_mon+1, utcdt->dt.tm_mday,
			utcdt->dt.tm_hour + (utcdt->dt.tm_min/60.0) + (utcdt->dt.tm_sec/3600.0),
			utcdt->cal == 0 ? 'j' : 'g', jdret);
	}
	/* else, is fresh from user input */
	else if (strlen(utcdt->tz) == 0) /* no tz given */
	{
		if (utcdt->offset != 0 || utcdt->isdst == 1) /* has offset given */
		{
			int sec = utcdt->dt.tm_sec;
			sec += utcdt->offset -
				((utcdt->isdst != -1 ? utcdt->isdst : 0) * 3600);
			i = swe_date_conversion(utcdt->dt.tm_year+1900, utcdt->dt.tm_mon+1, utcdt->dt.tm_mday,
				utcdt->dt.tm_hour + (utcdt->dt.tm_min/60.0) + (sec/3600.0),
				utcdt->cal == 0 ? 'j' : 'g', jdret);
		}
		else /* no offset, assume utc */
		{
			i = swe_date_conversion(utcdt->dt.tm_year+1900, utcdt->dt.tm_mon+1, utcdt->dt.tm_mday,
				utcdt->dt.tm_hour + (utcdt->dt.tm_min/60.0) + (utcdt->dt.tm_sec/3600.0),
				utcdt->cal == 0 ? 'j' : 'g', jdret);
		}
	}
	else /* has tz */
	{
		int sec = utcdt->dt.tm_sec;
		if (setenv("TZ", utcdt->tz, 1) != 0)
			return -1; /* invalid tz */
		if (mktime(&(utcdt->dt)) == -1) /* using user's offset and dst */
		{
			if (utcdt->isdst < 0)
				return -2; /* we need user input dst */
			sec += utcdt->offset - (utcdt->isdst * 3600);
			swe_date_conversion(utcdt->dt.tm_year+1900, utcdt->dt.tm_mon+1, utcdt->dt.tm_mday,
				utcdt->dt.tm_hour + (utcdt->dt.tm_min/60.0) + (sec/3600.0),
				utcdt->cal == 0 ? 'j' : 'g', jdret);
			i = -1; /* needs to be normalized */
		}
		else /* mktime succeeded */
		{
			if (utcdt->dt.tm_isdst == -1 && utcdt->isdst < 0)
				return -2; /* we need user input dst, dst switch */
			/* copy some time zone info */
			strcpy(utcdt->tzname, tzname[0]);
			utcdt->offset = timezone;
			if (daylight == 0) /* this tz never has dst */
				utcdt->isdst = 0;
			/* get jd */
			sec += utcdt->offset -
				((utcdt->dt.tm_isdst != -1 ? utcdt->dt.tm_isdst : utcdt->isdst) * 3600);
			swe_date_conversion(utcdt->dt.tm_year+1900, utcdt->dt.tm_mon+1, utcdt->dt.tm_mday,
				utcdt->dt.tm_hour + (utcdt->dt.tm_min/60.0) + (sec/3600.0),
				utcdt->cal == 0 ? 'j' : 'g', jdret);
			i = -1; /* needs to be normalized */
		}
		unsetenv("TZ");
	}
	if (i < 0) /* normalize dt */
	{
		int ret[6];
		swh_revjul(*jdret, utcdt->cal, ret);
		utcdt->dt.tm_year = ret[0] - 1900;
		utcdt->dt.tm_mon = ret[1] - 1;
		utcdt->dt.tm_mday = ret[2];
		utcdt->dt.tm_hour = ret[3];
		utcdt->dt.tm_min = ret[4];
		utcdt->dt.tm_sec = ret[5];
		utcdt->dt.tm_isdst = 0;
	}
	utcdt->status = 0;
	return 0;
}

/** @brief
*/
int swh_jd2UtcDt(double jd, swh_UtcDt *utcdt)
{
	assert(utcdt->cal == SE_GREG_CAL || utcdt->cal == SE_JUL_CAL);
	int ret[6];
	swh_revjul(jd, utcdt->cal, ret);
	struct tm *dt = &(utcdt->dt);
	dt->tm_year = ret[0] - 1900;
	dt->tm_mon = ret[1] -1 ;
	dt->tm_mday = ret[2];
	dt->tm_hour = ret[3];
	dt->tm_min = ret[4];
	dt->tm_sec = ret[5];
	return 0;
}


/** @brief Revert utc datetime to local datetime
**
** @param utcdt Utc datetime
** @param locdt Returned local datetime
** @return 0 if ok, or -1 is utcdt status is not 0
*/
int swh_UtcDt2local(const swh_UtcDt *utcdt, swh_UtcDt *locdt)
{
	if (utcdt->status != 0)
		return -1;
	if (utcdt->offset != 0 || utcdt->isdst == 1) /* has offset */
	{
		assert(utcdt->isdst != -1 || utcdt->dt.tm_isdst != -1);
		int i;
		double jdret;
		int ret[6], sec = utcdt->dt.tm_sec;
		sec -= utcdt->offset -
			((utcdt->isdst != -1 ? utcdt->isdst : utcdt->dt.tm_isdst) * 3600);
		i = swe_date_conversion(utcdt->dt.tm_year+1900, utcdt->dt.tm_mon+1, utcdt->dt.tm_mday,
			utcdt->dt.tm_hour + (utcdt->dt.tm_min/60.0) + (sec/3600.0),
			utcdt->cal == 0 ? 'j' : 'g', &jdret);
		swh_revjul(jdret, utcdt->cal, ret);
		locdt->dt.tm_year = ret[0] - 1900;
		locdt->dt.tm_mon = ret[1] - 1;
		locdt->dt.tm_mday = ret[2];
		locdt->dt.tm_hour = ret[3];
		locdt->dt.tm_min = ret[4];
		locdt->dt.tm_sec = ret[5];
		locdt->dt.tm_isdst = utcdt->dt.tm_isdst;
		strcpy(locdt->tz, utcdt->tz);
		strcpy(locdt->tzname, utcdt->tzname);
		locdt->cal = utcdt->cal;
		locdt->offset = utcdt->offset;
		locdt->isdst = utcdt->isdst;
	}
	else /* no offset */
	{
		*locdt = *utcdt;
	}
	locdt->status = -1;
	return 0;
}

/** @brief Get a string representation of UtcDt
**
** @param utcdt Utc datetime
** @param ret Returned string, declared as char[25]
** @return Number of char written, or negative on error
*/
int swh_sprintUtcDt(const swh_UtcDt *utcdt, char *ret)
{
	int i;
	if (strlen(utcdt->tzname) != 0)
		i = sprintf(ret, "%4d/%.2d/%.2d %.2d:%.2d:%.2d %s",
			utcdt->dt.tm_year+1900, utcdt->dt.tm_mon+1, utcdt->dt.tm_mday,
			utcdt->dt.tm_hour, utcdt->dt.tm_min, utcdt->dt.tm_sec, utcdt->tzname);
	else
		i = sprintf(ret, "%4d/%.2d/%.2d %.2d:%.2d:%.2d",
			utcdt->dt.tm_year+1900, utcdt->dt.tm_mon+1, utcdt->dt.tm_mday,
			utcdt->dt.tm_hour, utcdt->dt.tm_min, utcdt->dt.tm_sec);
	return i;
}

#ifdef __cplusplus
} /* extern "C" */
#endif
