/*
    Swephelp

    Copyright 2007-2009 Stanislas Marquis <stnsls@gmail.com>

    Swephelp is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    Swephelp is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Swephelp.  If not, see <http://www.gnu.org/licenses/>.
*/

/** @file swhdatetime.h
** @brief swephelp date and time functions
**
** @author Stanislas Marquis <stnsls@gmail.com>
** @date 11.01.2009
*/

#ifndef SWHDATETIME_H
#define SWHDATETIME_H

#ifndef SWH_USE_ALIASES
#define SWH_USE_ALIASES     1
#endif

#ifdef __cplusplus
extern "C"
{
#endif

#include <swephexp.h>
#include <time.h>
#include <stdio.h>

/** @brief Get current Julian day number, Gregorian calendar
** @return Julian day number
*/
double swh_jdnow(void);

/** @brief Get Julian day number from a date and time
**
** A conveniance function that, compared to swe_julday, does not require
** hour in decimal format, but accepts integers only.
**
** @attention Does not check date validity. See swe_date_conversion.
**
** @see swh_revjul()
**
** @param year Year (4 digits)
** @param mon Month [1;12]
** @param day Day [1;31]
** @param hour Hour [0;23]
** @param min Minutes [0;59]
** @param sec Seconds [0;59]
** @param flag Calendar type (SE_GREG_CAL|SE_JUL_CAL)
** @return Julian day number
*/
#define swh_julday(year, mon, day, hour, min, sec, flag)\
        swe_julday(year, mon, day, ((hour)+((min)/60.0)+((sec)/3600.0)), flag)

/** @brief Reverse Julian day to date and time
**
** Similar to swe_revjul, but returns time with three integers instead
** of one double. (Also tries to avoid some floating points rounding errors.)
**
** @see swh_julday()
**
** @param jd Julian day
** @param flag Calendar type (SE_GREG_CAL|SE_JUL_CAL)
** @param dt Results, declared as int[6] (year, month, day, hour, min, sec)
** @return 0
*/
int swh_revjul(double jd, int flag, int *dt);

/** @typedef swh_UtcDt
** @brief UTC datetime and information
**
** This struct represents a moment in time expressed as UTC, used inside some
** astrology software.
** It is able produce the julian day calculation fast, is easy to add or remove
** chunks of time, and remembers how the user entered it (timezone, utc offset
** and daylight savings).
*/
typedef struct
{
    struct tm dt; /* date and time. wday, yday are ignored. */
    int     cal; /* 0=julian, 1=gregorian */
    char    tz[35]; /* timezone (for tzset) */
    char    tzname[5]; /* tzname */
    int     offset; /* utc offset in seconds, without dst */
    int     isdst; /* dst 0=no 1=yes -1=undetermined */
    int     status; /* status -1=local 0=utc */
} swh_UtcDt;

/** @brief UtcDt constructor
**
** This will allocate memory for an UtcDt object and return it. Internal
** variables will be set to no timezone (empty strings), gregorian calendar,
** no offset and unknown daylight flag. User must fill in the tm struct with
** date and time information and possibly the timezone (given as zoneinfo file
** ex: ":Europe/Zurich" or any suitable for tzset) or the offset, eventually
** the isdst flag....
**
** @return new UtcDt
*/
swh_UtcDt *swh_newUtcDt(void);

/** @brief Get Julian day from given UtcDt
**
** Calculate Julian day for given UtcDt object. By the way, out of bound values
** are normalized.
**
** @return 0 if success, -1 on error, -2 needs user-defined dst
*/
int swh_mkUtcDt(swh_UtcDt *utcdt, double *jdret);

/** @brief Revert utc datetime to local datetime
**
** @param utcdt Utc datetime
** @param locdt Returned local datetime
** @return 0 if ok, or -1 is utcdt status is not 0
*/
int swh_UtcDt2local(const swh_UtcDt *utcdt, swh_UtcDt *locdt);

/** @brief Get a string representation of UtcDt
**
** @param utcdt Utc datetime
** @param ret Returned string, declared as char[25]
** @return Number of char written, or negative on error
*/
int swh_sprintUtcDt(const swh_UtcDt *utcdt, char *ret);

#if SWH_USE_ALIASES

#define DateTimeInfo        swh_UtcDt
#define DateTimeInfo_Create()       swh_newUtcDt()
#define DateTimeInfo_AsJulianDay(utcdt, jdret)      swh_mkUtcDt(utcdt, jdret)
#define DateTimeInfo_Localize(utcdt, locdt)     swh_UtcDt2local(utcdt, locdt)
#define DateTimeInfo_Print(utcdt, ret)      swh_sprintUtcDt(utcdt, ret)

#endif /* SWH_USE_ALIASES */


#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* swhdatetime.h */
