# -*- coding: utf-8 -*-
#------------------------------------------------------------------------------
# file: $Id: devinfo.py 27 2012-06-27 02:26:15Z griff1n $
# lib:  pysyncml.devinfo
# auth: griffin <griffin@uberdev.org>
# date: 2012/05/15
# copy: (C) CopyLoose 2012 UberDev <hardcore@uberdev.org>, No Rights Reserved.
#------------------------------------------------------------------------------

'''
The ``pysyncml.model.devinfo`` provides the device information sharing
classes.

For more information and links to documentation, please go to::

  http://pypi.python.org/pypi/pysyncml/

'''

import sys, logging, inspect
from elementtree import ElementTree as ET
from sqlalchemy import Column, Integer, Boolean, String, Text, ForeignKey
from sqlalchemy.orm import relation, synonym, backref
from sqlalchemy.orm.exc import NoResultFound
from .. import constants, common

log = logging.getLogger(__name__)

# TBD: DeviceInfo should implement something like items.base.Ext

#------------------------------------------------------------------------------
def notNoneOr(value, other):
  return value if value is not None else other

#------------------------------------------------------------------------------
def decorateModel(model):

  #----------------------------------------------------------------------------
  class DeviceInfo(model.DatabaseObject):
    # tbd: since adapter <=> device is one-to-one, shouldn't this be a primary key?...
    adapter_id        = Column(Integer, ForeignKey('%s_adapter.id' % (model.prefix,),
                                                   onupdate='CASCADE', ondelete='CASCADE'),
                               nullable=False, index=True)
    adapter           = relation('Adapter', backref=backref('_devinfo', uselist=False,
                                                            cascade='all, delete-orphan',
                                                            passive_deletes=True))
    devID             = Column(String(4095), nullable=False)
    devType           = Column(String(4095), nullable=False, default=constants.DEVTYPE_WORKSTATION)
    manufacturerName  = Column(String(4095), nullable=True, default='-')
    modelName         = Column(String(4095), nullable=True, default='-')
    oem               = Column(String(4095), nullable=True, default='-')
    hardwareVersion   = Column(String(255), nullable=True, default='-')
    firmwareVersion   = Column(String(255), nullable=True, default='-')
    softwareVersion   = Column(String(255), nullable=True, default='-')
    utc               = Column(Boolean, nullable=True, default=True)
    largeObjects      = Column(Boolean, nullable=True, default=True)
    hierarchicalSync  = Column(Boolean, nullable=True, default=True)
    numberOfChanges   = Column(Boolean, nullable=True, default=True)

    strAttributeMap = [
      ('manufacturerName',    'Man'),
      ('modelName',           'Mod'),
      ('oem',                 'OEM'),
      ('firmwareVersion',     'FwV'),
      ('softwareVersion',     'SwV'),
      ('hardwareVersion',     'HwV'),
      ('devID',               'DevID'),
      ('devType',             'DevTyp'),
      ]

    boolAttributeMap = [
      ('utc',                 'UTC'),
      ('largeObjects',        'SupportLargeObjs'),
      ('hierarchicalSync',    'SupportHierarchicalSync'),
      ('numberOfChanges',     'SupportNumberOfChanges'),
      ]

    #--------------------------------------------------------------------------
    def __init__(self, *args, **kw):
      # TBD: why on *EARTH* do i have to do this?...
      self._setDefaults()
      super(DeviceInfo, self).__init__(*args, **kw)

    #----------------------------------------------------------------------------
    def __repr__(self):
      ret = '<Device "%s": devType=%s' % (self.devID, self.devType)
      for attr in self.__table__.c.keys():
        if attr in self.__syscols__ \
           or attr in ('devID', 'devType', 'adapter_id',) \
           or getattr(self, attr) is None:
          continue
        ret += '; %s=%s' % (attr, str(getattr(self, attr)))
      return ret + '>'

    # #----------------------------------------------------------------------------
    # def save(self, adapter):
    #   if adapter._db.device is None:
    #     adapter._db.device = adapter.model.Device()
    #   self._db = adapter._db.device
    #   for attr in common.dbattrs(adapter.model.Device, DeviceInfo):
    #     setattr(self._db, attr, getattr(self, attr))

    # #----------------------------------------------------------------------------
    # @staticmethod
    # def load(adapter):
    #   if adapter._db.device is None:
    #     return None
    #   ret = DeviceInfo()
    #   ret._db = adapter._db.device
    #   for attr in common.dbattrs(adapter.model.Device, DeviceInfo):
    #     setattr(ret, attr, getattr(ret._db, attr))
    #   return ret

    #----------------------------------------------------------------------------
    def toSyncML(self, dtdVersion, stores):
      if dtdVersion is None:
        dtdVersion = constants.SYNCML_DTD_VERSION_1_2
      if dtdVersion != constants.SYNCML_DTD_VERSION_1_2:
        raise common.InternalError('unsupported DTD version "%s"' % (dtdVersion,))
      xret = ET.Element('DevInf', {'xmlns': constants.NAMESPACE_DEVINF})
      ET.SubElement(xret, 'VerDTD').text = dtdVersion
      for attr, xname in DeviceInfo.strAttributeMap:
        # tbd: should i enforce the fact that these are all *required*?...
        if getattr(self, attr) is not None:
          ET.SubElement(xret, xname).text = getattr(self, attr)
      for attr, xname in DeviceInfo.boolAttributeMap:
        if getattr(self, attr) is not None and getattr(self, attr):
          ET.SubElement(xret, xname)
      for store in stores or []:
        xret.append(store.toSyncML())
      return xret

    #----------------------------------------------------------------------------
    @staticmethod
    def fromSyncML(xnode):
      # tbd: it would be *great* if i could delete the namespacing here...
      devinfo   = DeviceInfo()
      stores    = []
      dtdVersion = xnode.findtext('VerDTD')
      if dtdVersion != constants.SYNCML_DTD_VERSION_1_2:
        raise common.ProtocolError('unsupported DTD version "%s"' % (dtdVersion,))
      for attr, xname in DeviceInfo.strAttributeMap:
        setattr(devinfo, attr, xnode.findtext(xname))
      for attr, xname in DeviceInfo.boolAttributeMap:
        setattr(devinfo, attr, xnode.find(xname) is not None)
      for child in xnode:
        if child.tag != 'DataStore':
          continue
        stores.append(model.Store.fromSyncML(child))
      return (devinfo, stores)

  model.DeviceInfo = DeviceInfo

#------------------------------------------------------------------------------
# end of $Id: devinfo.py 27 2012-06-27 02:26:15Z griff1n $
#------------------------------------------------------------------------------
