# -*- coding: utf-8 -*-
#------------------------------------------------------------------------------
# file: $Id: adapter.py 34 2012-07-03 02:48:00Z griff1n $
# lib:  pysyncml.adapter
# auth: griffin <griffin@uberdev.org>
# date: 2012/05/19
# copy: (C) CopyLoose 2012 UberDev <hardcore@uberdev.org>, No Rights Reserved.
#------------------------------------------------------------------------------

'''
The ``pysyncml.model.adapter`` package exposes the Adapter implementation of
the pysyncml package.
'''

import sys, os, time, logging, urllib2, cookielib
from sqlalchemy import orm
from sqlalchemy import Column, Integer, Boolean, String, Text, ForeignKey
from sqlalchemy.orm import relation, synonym, backref
from .. import common, constants, codec, state

log = logging.getLogger(__name__)

# TODO: the current algorithm for finding the "local" adapter is to search
#       for adapters where isLocal == False... i should move to one-to-many
#       parent-child relationships and search for adapters where parent
#       == None...

#------------------------------------------------------------------------------
def decorateModel(model):

  #----------------------------------------------------------------------------
  class Adapter(model.DatabaseObject):
    devID             = Column(String(4095), nullable=False, index=True)
    name              = Column(String(4095), nullable=True)
    isLocal           = Column(Boolean, nullable=True)
    createdDate       = Column(Integer, default=common.ts)
    isServer          = Column(Boolean, nullable=True)
    url               = Column(String(4095), nullable=True)
    auth              = Column(Integer, nullable=True)
    username          = Column(String(4095), nullable=True)
    password          = Column(String(4095), nullable=True)
    lastSessionID     = Column(Integer)
    firstSync         = Column(Integer)
    lastSync          = Column(Integer)
    maxGuidSize       = Column(Integer)
    maxMsgSize        = Column(Integer)
    maxObjSize        = Column(Integer)
    _peer             = None

    @property
    def devinfo(self):
      return self._devinfo
    @devinfo.setter
    def devinfo(self, devinfo):
      if devinfo is self._devinfo:
        return
      self._devinfo = devinfo
      if self.devID is None:
        self.devID = devinfo.devID
      if self.devID is not None:
        self._context._model.session.flush()

    @property
    def stores(self):
      return dict((store.uri, store) for store in self._stores)

    @property
    def peer(self):
      return self._peer
    @peer.setter
    def peer(self, peer):
      if peer is self._peer:
        return
      if peer is not None and peer.id is None:
        model.session.add(peer)
      self._peer = peer

    #--------------------------------------------------------------------------
    def __init__(self, *args, **kw):
      if 'isLocal' not in kw:
        kw['isLocal'] = kw.get('url') is None
      if 'devID' not in kw:
        kw['devID'] = kw.get('url')
      # TODO: why on *EARTH* do i have to do this?...
      self._setDefaults()
      super(Adapter, self).__init__(*args, **kw)
      self._initHelpers()

    #--------------------------------------------------------------------------
    @orm.reconstructor
    def __dbinit__(self):
      self._initHelpers()

    #--------------------------------------------------------------------------
    def _initHelpers(self):
      if not self.isLocal:
        self._ckjar    = cookielib.CookieJar()
        # todo: should i be using the default opener instead?...
        #       i think the only reason that a custom opener is useful is to
        #       be able to add cookie handling - but can't that be done with
        #       the default opener as well?...
        self._opener   = urllib2.OpenerDirector()
        self._opener.add_handler(urllib2.HTTPHandler())
        self._opener.add_handler(urllib2.HTTPSHandler())
        self._opener.add_handler(urllib2.HTTPCookieProcessor(self._ckjar))

    #--------------------------------------------------------------------------
    def cleanUri(self, uri):
      return os.path.normpath(uri)

    #--------------------------------------------------------------------------
    def getKnownPeers(self):
      return model.Adapter.q(isLocal=False).all()

    #--------------------------------------------------------------------------
    def addStore(self, store):
      store.uri = self.cleanUri(store.uri)
      for curstore in self._stores:
        if curstore.uri == store.uri:
          curstore.merge(store)
          return curstore
      self._stores.append(store)
      if self.devID is not None:
        self._context._model.session.flush()
      return self._stores[-1]

    #--------------------------------------------------------------------------
    def _dbsave(self):
      # if model.context.autoCommit:
      #   model.session.commit()
      model.context.save()

    #--------------------------------------------------------------------------
    def sync(self, mode=constants.SYNCTYPE_AUTO):
      # # todo: deal with this paranoia... perhaps move this into the synchronizer?...
      # self._context._model.session.flush()
      # remap mode from SYNCTYPE_ to ALERT_
      if mode is not None:
        omode = mode
        mode  = common.synctype2alert(mode)
        if mode is None:
          raise TypeError('unknown/unsupported sync mode "%r"' % (omode,))
      if self.devinfo is None:
        raise common.InvalidAdapter('no device info provided')
      if self.peer is None:
        raise common.InvalidAdapter('no peer registered')
      log.debug('starting %s sync with peer "%s"', common.mode2string(mode), self.peer.devID)

      session = state.Session(
        id         = ( self.peer.lastSessionID or 0 ) + 1,
        isServer   = False,
        mode       = mode,
        )

      for store in self.stores.values():
        if store.agent is None:
          continue
        peerUri = self.router.getTargetUri(store.uri, mustExist=False)
        if peerUri is None:
          continue
        ds = common.adict(
          # TODO: perhaps share this "constructor" with router/protocol?...
          lastAnchor = self.peer.stores[peerUri].binding.sourceAnchor,
          nextAnchor = str(int(time.time())),
          mode       = mode,
          action     = 'alert',
          peerUri    = peerUri,
          stats      = state.Stats(),
          )

        log.critical('TODO: implement auto-detect mode')
        # TODO: perhaps this should be pushed into synchronizer? it
        #       should be able to perform more in-depth logic.

        # if ds.mode is None:
        #   if agent.getChanges() is None:
        #     ds.mode = constants.ALERT_SLOW_SYNC
        #   else:
        #     # todo: anything smarter that should be done?...
        #     ds.mode = constants.ALERT_TWO_WAY
        # if ds.lastAnchor is None:
        #   if ds.mode in [
        #     constants.ALERT_SLOW_SYNC,
        #     constants.ALERT_REFRESH_FROM_CLIENT,
        #     constants.ALERT_REFRESH_FROM_SERVER,
        #     ]:
        #     pass
        #   elif ds.mode in [
        #     constants.ALERT_TWO_WAY,
        #     constants.ALERT_ONE_WAY_FROM_CLIENT,
        #     constants.ALERT_ONE_WAY_FROM_SERVER,
        #     ]:
        #     log.info('forcing slow-sync for datastore "%s" (no previous successful synchronization)', uri)
        #     ds.mode = constants.ALERT_SLOW_SYNC
        #   else:
        #     raise common.ProtocolError('unexpected sync mode "%d" requested' % (ds.mode,))

        session.dsstates[store.uri] = ds

      commands = self.protocol.initialize(self, session)

      self._transmit(session, commands)
      self._dbsave()
      return self._session2stats(session)

    #--------------------------------------------------------------------------
    def _session2stats(self, session):
      ret = common.adict()
      for uri, ds in session.dsstates.items():
        stats = ds.stats
        stats.mode = common.alert2synctype(ds.mode)
        ret[uri] = stats
      log.info('session statistics: %r', ret)
      return ret

    #--------------------------------------------------------------------------
    def _transmit(self, session, commands, response=None):

      commands = self.protocol.negotiate(self, session, commands)

      if not session.isServer \
         and len(commands) == 3 \
         and commands[0].cmd == constants.CMD_SYNCHDR \
         and commands[1].cmd == constants.CMD_STATUS \
         and commands[1].statusOf == constants.CMD_SYNCHDR \
         and commands[1].statusCode == str(constants.STATUS_OK) \
         and commands[2].cmd == constants.CMD_FINAL:
        for uri, ds in session.dsstates.items():
          log.debug('storing next anchor here="%s", peer="%s" for URI "%s"',
                    ds.nextAnchor, ds.peerNextAnchor, uri)
          self.peer.stores[ds.peerUri].binding.sourceAnchor = ds.nextAnchor
          self.peer.stores[ds.peerUri].binding.targetAnchor = ds.peerNextAnchor
        self.peer.lastSessionID = session.id
        log.debug('synchronization complete for "%s" (s%s.m%s)',
                  self.peer.devID, session.id, session.lastMsgID)
        return

      request = state.Request(
        commands    = commands,
        contentType = None,
        body        = None,
        )

      xtree = self.protocol.commands2tree(self, session, commands)
      (request.contentType, request.body) = self.codec.encode(xtree)

      # update the session with the last request commands so that
      # when we receive the response package, it can be compared against
      # that.
      # TODO: should that only be done on successful transmit?...
      session.lastCommands = commands
      if response is None:
        self.peer.handleRequest(session, request, adapter=self)
      else:
        response.contentType = request.contentType
        response.body        = request.body

    #--------------------------------------------------------------------------
    def handleRequest(self, session, request, response=None, adapter=None):
      # # todo: deal with this paranoia... perhaps move this into the synchronizer?...
      # self._context._model.session.commit()
      if self.isLocal:
        return self._handleRequestLocal(session, request, response)
      return self._handleRequestRemote(session, request, adapter)

    #--------------------------------------------------------------------------
    def _handleRequestLocal(self, session, request, response=None):
      commands = self._receive(session, request) or []
      log.debug('beginning negotiation of device "%s" (s%d.m%d)',
                self.peer.devID, session.id, session.msgID)
      if session.msgID > 20:
        log.error('too many client/server messages, pending commands: %r', commands)
        raise common.ProtocolError('too many client/server messages')
      self._transmit(session, commands, response)
      if session.isServer:
        self._dbsave()
        return self._session2stats(session)

    #----------------------------------------------------------------------------
    def _handleRequestRemote(self, session, request, adapter):
      # TODO: should this be broken out into a separate sub-class?...
      req = urllib2.Request(session.respUri or self.url, request.body)
      req.add_header('content-type', request.contentType or 'application/vnd.syncml+xml')
      req.add_header('x-syncml-client', 'pysyncml/' + common.versionString)
      # todo: add any other syncml headers?...
      # # ***********************************************************************
      # # TODO: **HACKALERT** **HACKALERT** **HACKALERT** **HACKALERT** **HACKALERT**
      # #      remove this HACK!...
      # req.adapter = adapter
      # req.session = session
      # # ***********************************************************************
      res = self._opener.open(req, request.body)
      # TODO: check response status...
      res = state.Request(body=res.read(), headers=res.info().headers)
      res.headers = [map(lambda x: x.strip(), h.split(':', 1))
                     for h in res.headers]
      res.headers = dict([(k.lower(), v) for k, v in res.headers])
      adapter.handleRequest(session, res)

    #--------------------------------------------------------------------------
    def _receive(self, session, request):
      if not session.isServer:
        session.lastMsgID = session.msgID
        session.nextMsgID
      else:
        session.lastCommands = session.lastCommands or []
      xtree = codec.Codec.autoDecode(request.headers['content-type'], request.body)
      return self.protocol.tree2commands(self, session, session.lastCommands, xtree)

    # # #----------------------------------------------------------------------------
    # # def describe(self, stream):
    # #   s2 = common.IndentStream(stream)
    # #   s3 = common.IndentStream(s2)
    # #   stream.write('Local device:\n')
    # #   print >>s2, 'Device ID:', self.devinfo.devID
    # #   if len(self.agents) <= 0:
    # #     print >>s2, 'DataStores: N/A'
    # #   else:
    # #     print >>s2, 'DataStores:'
    # #     for agent in self.agents.values():
    # #       agent.dsinfo.describe(s3)
    # #   stream.write('Remote device:\n')
    # #   print >>s2, 'Device ID:', self.target.devID
    # #   if self.target.dsinfos is None:
    # #     print >>s2, 'DataStores: N/A'
    # #   else:
    # #     print >>s2, 'DataStores:'
    # #     for dsi in self.target.dsinfos:
    # #       dsi.describe(s3)
    # #   if len(self.router.routes) <= 0:
    # #     stream.write('Sync routing: N/A\n')
    # #   else:
    # #     stream.write('Sync routing:\n')
    # #     for route in self.router.routes.values():
    # #       anchors = self.getLastAnchorSet(route.sourceUri)
    # #       print >>s2, '%s <=> %s%s, anchors: %s/%s' \
    # #             % (route.sourceUri, route.targetUri,
    # #                ' (auto)' if route.autoMapped else '',
    # #                anchors[0] or '-', anchors[1] or '-')

  model.Adapter = Adapter

#------------------------------------------------------------------------------
# end of $Id: adapter.py 34 2012-07-03 02:48:00Z griff1n $
#------------------------------------------------------------------------------
