# -*- coding: utf-8 -*-
#------------------------------------------------------------------------------
# file: $Id: store.py 34 2012-07-03 02:48:00Z griff1n $
# lib:  pysyncml.store
# auth: griffin <griffin@uberdev.org>
# date: 2012/06/14
# copy: (C) CopyLoose 2012 UberDev <hardcore@uberdev.org>, No Rights Reserved.
#------------------------------------------------------------------------------

'''
The ``pysyncml.model.store`` provides a SyncML datastore abstraction
via the :class:`pysyncml.model.store.Store` class, which includes both
the datastore meta information and, if the datastore is local, an
agent to execute data interactions.
'''

import sys, json, logging
from elementtree import ElementTree as ET
from sqlalchemy import Column, Integer, Boolean, String, Text, ForeignKey
from sqlalchemy.orm import relation, synonym, backref
from .. import common, constants, ctype

log = logging.getLogger(__name__)

#------------------------------------------------------------------------------
def decorateModel(model):

  #----------------------------------------------------------------------------
  class Store(model.DatabaseObject):

    allSyncTypes = [
      constants.SYNCTYPE_TWO_WAY,
      constants.SYNCTYPE_SLOW_SYNC,
      constants.SYNCTYPE_ONE_WAY_FROM_CLIENT,
      constants.SYNCTYPE_REFRESH_FROM_CLIENT,
      constants.SYNCTYPE_ONE_WAY_FROM_SERVER,
      constants.SYNCTYPE_REFRESH_FROM_SERVER,
      constants.SYNCTYPE_SERVER_ALERTED,
      ]

    adapter_id        = Column(Integer, ForeignKey('%s_adapter.id' % (model.prefix,),
                                                   onupdate='CASCADE', ondelete='CASCADE'),
                               nullable=False, index=True)
    adapter           = relation('Adapter', backref=backref('_stores', # order_by=id,
                                                            cascade='all, delete-orphan',
                                                            passive_deletes=True))
    uri               = Column(String(4095), nullable=False, index=True)
    displayName       = Column(String(4095))
    _syncTypes        = Column('syncTypes', String(4095)) # note: default set in __init__
    maxGuidSize       = Column(Integer)                   # note: default set in __init__
    maxObjSize        = Column(Integer)                   # note: default set in __init__
    agent             = None

    #  binding       = None

    @property
    def syncTypes(self):
      return json.loads(self._syncTypes or 'null')
    @syncTypes.setter
    def syncTypes(self, types):
      self._syncTypes = json.dumps(types)

    @property
    def contentTypes(self):
      if self.agent is not None:
        return self.agent.contentTypes
      return self._contentTypes

    @property
    def peer(self):
      return self.getPeerStore()

    #--------------------------------------------------------------------------
    def getPeerStore(self, adapter=None):
      if not self.adapter.isLocal:
        if adapter is None:
          # todo: implement this...
          raise common.InternalError('local adapter is required for call to remoteStore.getPeerStore()')
        uri = adapter.router.getSourceUri(self.uri, mustExist=False)
        if uri is None:
          return None
        return adapter.stores[uri]
      if self.adapter.peer is None:
        return None
      ruri = self.adapter.router.getTargetUri(self.uri, mustExist=False)
      if ruri is None:
        return None
      return self.adapter.peer.stores[ruri]

    #--------------------------------------------------------------------------
    def __init__(self, **kw):
      # TODO: this is a little hack... it is because the .merge() will
      #       otherwise override valid values with null values when the merged-in
      #       store has not been flushed, and because this is a valid value,
      #       open flush, is being nullified. ugh.
      # NOTE: the default is set here, not in the Column() definition, so that
      #       NULL values remain NULL during a flush) - since they are valid.
      self._syncTypes  = kw.get('syncTypes',   repr(Store.allSyncTypes))
      self.maxGuidSize = kw.get('maxGuidSize', common.getIntSize())
      self.maxObjSize  = kw.get('maxObjSize',  sys.maxint)
      super(Store, self).__init__(**kw)

    #----------------------------------------------------------------------------
    def __repr__(self):
      ret = '<Store "%s": uri=%s' % (self.displayName or self.uri, self.uri)
      if self.maxGuidSize is not None:
        ret += '; maxGuidSize=%d' % (self.maxGuidSize,)
      if self.maxObjSize is not None:
        ret += '; maxObjSize=%d' % (self.maxObjSize,)
      if self.syncTypes is not None and len(self.syncTypes) > 0:
        ret += '; syncTypes=%s' % (','.join([str(st) for st in self.syncTypes]),)
      if self.contentTypes is not None and len(self.contentTypes) > 0:
        ret += '; contentTypes=%s' % (','.join([str(ct) for ct in self.contentTypes]),)
      return ret + '>'

    #----------------------------------------------------------------------------
    def merge(self, store):
      if self.uri != store.uri:
        raise common.InternalError('unexpected merging of stores with different URIs (%s != %s)'
                                   % (self.uri, store.uri))
      self.displayName   = store.displayName
      self._contentTypes = store._contentTypes
      self.syncTypes     = store.syncTypes
      self.maxGuidSize   = store.maxGuidSize
      self.maxObjSize    = store.maxObjSize
      self.agent         = store.agent

      # TODO: what about bindings?...
      log.critical('TODO: handle bindings during Store.merge()...')

      return self

    #----------------------------------------------------------------------------
    def clearChanges(self):
      if self.adapter.isLocal:
        # TODO: THIS NEEDS TO BE SIGNIFICANTLY OPTIMIZED!... either:
        #         a) optimize this reverse lookup, or
        #         b) use a query that targets exactly the set of stores needed
        #       note that a pre-emptive model.session.flush() may be necessary.
        for peer in self.adapter.getKnownPeers():
          for store in peer._stores:
            if store.binding is not None and store.binding.uri == self.uri:
              store.clearChanges()
        return
      if self.id is None:
        model.session.flush()
      model.Change.q(store_id=self.id).delete()

    #----------------------------------------------------------------------------
    def registerChange(self, itemID, state):
      if self.adapter.isLocal:
        # TODO: THIS NEEDS TO BE SIGNIFICANTLY OPTIMIZED!... either:
        #         a) optimize this reverse lookup, or
        #         b) use a query that targets exactly the set of stores needed
        #       note that a pre-emptive model.session.flush() may be necessary.
        for peer in self.adapter.getKnownPeers():
          for store in peer._stores:
            if store.binding is not None and store.binding.uri == self.uri:
              store.registerChange(itemID, state)
        return
      if self.id is None:
        model.session.flush()
      itemID = str(itemID)
      model.Change.q(store_id=self.id, itemID=itemID).delete()
      change = model.Change(store_id=self.id, itemID=itemID, state=state)
      model.session.add(change)

    #--------------------------------------------------------------------------
    def getRegisteredChanges(self):
      return model.Change.q(store_id=self.id)

    # # #----------------------------------------------------------------------------
    # # def describe(self, s1):
    # #   s2 = common.IndentStream(s1)
    # #   print >>s1, self.displayName
    # #   print >>s2, 'URI:', self.uri
    # #   print >>s2, 'Sync types:', ','.join([str(e) for e in self.syncTypes or []])
    # #   print >>s2, 'Max ID size:', self.maxGuidSize or '-'
    # #   print >>s2, 'Max object size:', self.maxObjSize or '-'
    # #   print >>s2, 'Capabilities:'
    # #   for cti in self.contentTypes or []:
    # #     cti.describe(common.IndentStream(s2))

    #----------------------------------------------------------------------------
    def toSyncML(self):
      xstore = ET.Element('DataStore')
      if self.uri is not None:
        ET.SubElement(xstore, 'SourceRef').text = self.uri
      if self.displayName is not None:
        ET.SubElement(xstore, 'DisplayName').text = self.displayName
      if self.maxGuidSize is not None:
        # todo: this should ONLY be sent by the client... (according to the
        #       spec, but not according to funambol behavior...)
        ET.SubElement(xstore, 'MaxGUIDSize').text = str(self.maxGuidSize)
      if self.maxObjSize is not None:
        ET.SubElement(xstore, 'MaxObjSize').text = str(self.maxObjSize)
      if self.contentTypes is not None:
        rxpref = [ct for ct in self.contentTypes if ct.receive and ct.preferred]
        if len(rxpref) > 1:
          raise common.InvalidAgent('agents can prefer at most one rx content-type, not %r' % (rxpref,))
        if len(rxpref) == 1:
          xstore.append(rxpref[0].toSyncML('Rx-Pref'))
        for rx in [ct for ct in self.contentTypes if ct.receive and not ct.preferred]:
          xstore.append(rx.toSyncML('Rx'))
        txpref = [ct for ct in self.contentTypes if ct.transmit and ct.preferred]
        if len(txpref) > 1:
          raise common.InvalidAgent('agents can prefer at most one tx content-type, not %r' % (txpref,))
        if len(txpref) == 1:
          xstore.append(txpref[0].toSyncML('Tx-Pref'))
        for tx in [ct for ct in self.contentTypes if ct.transmit and not ct.preferred]:
          xstore.append(tx.toSyncML('Tx'))
      if self.syncTypes is not None and len(self.syncTypes) > 0:
        xcap = ET.SubElement(xstore, 'SyncCap')
        for st in self.syncTypes:
          ET.SubElement(xcap, 'SyncType').text = str(st)
      return xstore

    #----------------------------------------------------------------------------
    @staticmethod
    def fromSyncML(xnode):
      store = model.Store()
      store.uri = xnode.findtext('SourceRef')
      store.displayName = xnode.findtext('DisplayName')
      store.maxGuidSize = xnode.findtext('MaxGUIDSize')
      if store.maxGuidSize is not None:
        store.maxGuidSize = int(store.maxGuidSize)
      store.maxObjSize  = xnode.findtext('MaxObjSize')
      if store.maxObjSize is not None:
        store.maxObjSize = int(store.maxObjSize)
      store.syncTypes = [int(x.text) for x in xnode.findall('SyncCap/SyncType')]
      store._contentTypes = []
      for child in xnode:
        if child.tag not in ('Tx-Pref', 'Tx', 'Rx-Pref', 'Rx'):
          continue
        cti = model.ContentTypeInfo.fromSyncML(child)
        for curcti in store._contentTypes:
          if curcti.merge(cti):
            break
        else:
          store._contentTypes.append(cti)
      return store

  #----------------------------------------------------------------------------
  class ContentTypeInfo(model.DatabaseObject, ctype.ContentTypeInfoMixIn):
    store_id          = Column(Integer, ForeignKey('%s_store.id' % (model.prefix,),
                                                   onupdate='CASCADE', ondelete='CASCADE'),
                               nullable=False, index=True)
    store             = relation('Store', backref=backref('_contentTypes', # order_by=id,
                                                          cascade='all, delete-orphan',
                                                          passive_deletes=True))
    ctype             = Column(String(4095))
    _versions         = Column('versions', String(4095))
    preferred         = Column(Boolean, default=False)
    transmit          = Column(Boolean, default=True)
    receive           = Column(Boolean, default=True)

    @property
    def versions(self):
      return json.loads(self._versions or 'null')
    @versions.setter
    def versions(self, types):
      self._versions = json.dumps(types)

    def __str__(self):
      return ctype.ContentTypeInfoMixIn.__str__(self)

    def __repr__(self):
      return ctype.ContentTypeInfoMixIn.__repr__(self)

  #----------------------------------------------------------------------------
  class Binding(model.DatabaseObject):
    # todo: since store <=> binding is one-to-one, shouldn't this be a primary key?...
    store_id          = Column(Integer, ForeignKey('%s_store.id' % (model.prefix,),
                                                   onupdate='CASCADE', ondelete='CASCADE'),
                               nullable=False, index=True)
    targetStore       = relation('Store', backref=backref('binding', uselist=False,
                                                          cascade='all, delete-orphan',
                                                          passive_deletes=True))
    # todo: this uri *could* be replaced by an actual reference to the Store object...
    #       and then the getSourceStore() method can go away...
    #       *BUT* this would require a one-to-many Adapter<=>Adapter relationship...
    uri               = Column(String(4095), nullable=True)
    autoMapped        = Column(Boolean)
    sourceAnchor      = Column(String(4095), nullable=True)
    targetAnchor      = Column(String(4095), nullable=True)

    def getSourceStore(self, adapter):
      return adapter.stores[self.uri]

  #----------------------------------------------------------------------------
  class Change(model.DatabaseObject):
    store_id          = Column(Integer, ForeignKey('%s_store.id' % (model.prefix,),
                                                   onupdate='CASCADE', ondelete='CASCADE'),
                               nullable=False, index=True)
    # store             = relation('Store', backref=backref('changes',
    #                                                       cascade='all, delete-orphan',
    #                                                       passive_deletes=True))
    itemID            = Column(String(4095), index=True, nullable=False)
    state             = Column(Integer)
    updatedDate       = Column(Integer, default=common.ts)

  model.Store           = Store
  model.ContentTypeInfo = ContentTypeInfo
  model.Binding         = Binding
  model.Change          = Change

#------------------------------------------------------------------------------
# end of $Id: store.py 34 2012-07-03 02:48:00Z griff1n $
#------------------------------------------------------------------------------
