import time
import random
import string
import simplejson
import warnings
try:
    import requests
except ImportError:
    raise Exception('python-bada-push: library "requests" is required. Use "pip install requests"')
from requests.compat import urlencode

STATUS_CODE_SUCCESS = 1000 # see http://developer.bada.com/article/push-messaging-guide#status_codes_messages for details

class BadaPushError(Exception):
    '''
    General error: error of loading response from JSON, invalid HTTP response code, etc.
    '''

class ParameterError(BadaPushError):
    '''
    Argument value error that can be detected without server querying.
    '''

class ServerResponseError(BadaPushError):
    '''
    HTTP response is correct, but API says that something is wrong.
    '''
    def __init__(self, code, message):
        self.code = code
        super(ServerResponseError, self).__init__(message)



BADA_PUSH_SERVERS = { # [regid_prefix] = (Region_name, push_server_URL)
    '00': ('US East','https://useast.push.samsungosp.com:8088/spp/pns/api/push'),
    '01': ('US West','https://uswest.push.samsungosp.com:8088/spp/pns/api/push'),
    '02': ('Asia Pacific Southeast','https://apsoutheast.push.samsungosp.com:8088/spp/pns/api/push'),
    '03': ('EU West','https://euwest.push.samsungosp.com:8088/spp/pns/api/push'),
    '04': ('Asia Pacific Northeast','https://apnortheast.push.samsungosp.com:8088/spp/pns/api/push'),
    '05': ('Korea','https://apkorea.push.samsungosp.com:8088/spp/pns/api/push'),
    '06': ('China','https://apchina.push.samsungosp.com.cn:8088/spp/pns/api/push')
}

class PushMessage(object):
    def __init__(self, app_id, app_secret, reg_id, app_data,
                 request_id=None, action='SILENT', alert_message=None, badge_option=None, badge_number=None,
                 verify_ssl=True, url=None):
        '''
        http://developer.bada.com/article/push-messaging-guide
        @param url:           The 'region' within the curly bracket in the push request URL indicates where the bada Server is located and is decided by first 2 digits of regID.
                              00     US East     https://useast.push.samsungosp.com:8088/spp/pns/api/push
                              01     US West     https://uswest.push.samsungosp.com:8088/spp/pns/api/push
                              02     Asia Pacific Southeast     https://apsoutheast.push.samsungosp.com:8088/spp/pns/api/push
                              03     EU West     https://euwest.push.samsungosp.com:8088/spp/pns/api/push
                              04     Asia Pacific Northeast     https://apnortheast.push.samsungosp.com:8088/spp/pns/api/push
                              05     Korea     https://apkorea.push.samsungosp.com:8088/spp/pns/api/push
                              06     China     https://apchina.push.samsungosp.com.cn:8088/spp/pns/api/push
        @param app_id:        Issued by the bada Server for the purpose of application authentication.
                              It can be found in your application's manifest.xml file.
        @param app_secret:    Issued by the bada Server for the purpose of application authentication.
                              It can be found in your application's manifest.xml file.
        @param reg_id:        A unique identifier that represents a specific application on a specific device.
                              Your application on the bada device can receive a registration ID.
        @param app_data:      Message body (payload)
        @param request_id:    A unique value that distinguishes a push message. Autogenerated by default.
        @param action:        "ALERT": Store the message and alert the user.
                              "SILENT": Store the message without alerting the user.
                              "DISCARD": Discard the message.
        @param alert_message: Alert message shown to the user (if 'action' is not set as 'ALERT', this
                              value is not meaningful.) . Up to 127 bytes
        @param badge_option:  "INCREASE": Increase the badge number by the given value.
                              "DECREASE": Decrease the badge number by the given value.
                              "SET": Set badge number to the given value.
        @param badge_number:  0 ~ 999 if badge_option="SET"
        @verify_ssl:          set False for get rid of Error: [Errno 1] _ssl.c:490: error:14090086:SSL routines:SSL3_GET_SERVER_CERTIFICATE:certificate verify failed
        @raise ParameterError: on invalid parameters
        '''
        if url:
            warnings.warn('bada_push.PushMessage: parameter url is deprecated. Urls now hardcoded in bada_push')
            self.url = url
        else:
            if reg_id[:2] in BADA_PUSH_SERVERS:
                self.url = BADA_PUSH_SERVERS[reg_id[:2]][1]
            else:
                raise ParameterError('First 2 digits of regID (%s) incorrect. Can not find server for it.' % reg_id[:2])
        self.app_id = app_id
        self.app_secret = app_secret
        self.reg_id = reg_id
        self.app_data = app_data
        # generate random request_id
        self.request_id = request_id or '%s-%s' % (time.time(), ''.join([random.choice(string.ascii_lowercase) for i in range(10)])) #@UnusedVariable
        if action not in ('ALERT', 'SILENT', 'DISCARD'):
            raise ParameterError('Action %s is invalid' % action)
        self.action = action
        if alert_message is not None and len(alert_message) > 127:
            raise ParameterError('Alert message too long')

        self.alert_message = alert_message
        if badge_option is not None and badge_option not in ('INCREASE', 'DECREASE', 'SET'):
            raise ParameterError('Badge_option %s is invalid' % badge_option)
        self.badge_option = badge_option
        self.badge_number = badge_number
        self.verify_ssl = verify_ssl

    def send(self):
        '''
        Send query to server.
        Return tuple (requests, response) in string.
        @raise BadaPushError: something wrong
        @raise ServerResponseError: API says that something is wrong
        @raise requests.exceptions.RequestException: any network level error
        '''
        # construct request
        headers = {'appID':self.app_id, 'appSecret':self.app_secret}
        message = {'action':self.action}
        if self.alert_message:
            message['alertMessage'] = self.alert_message
        if self.badge_option:
            message['badgeOption'] = self.badge_number
        if self.badge_number:
            message['badgeNumber'] = self.badge_number
        body = {'regID': self.reg_id,
                'requestID': self.request_id,
                'message':urlencode(message),
                'appData':self.app_data}

        # get response
        request_serialized = simplejson.dumps(body, ensure_ascii=False).encode('utf8')
        response = requests.post(self.url, data=request_serialized, headers=headers, verify=self.verify_ssl)

        # parse response
        if not response.ok:
            raise BadaPushError('HTTP response code is invalid: %r' % response)
        try:
            response_in_json = simplejson.loads(response.content)
        except:
            raise BadaPushError('Could not parse response: %r' % response.content)
        if response_in_json['results'][0]['statusCode'] != STATUS_CODE_SUCCESS:
            raise ServerResponseError(
                response_in_json['results'][0]['statusCode'],
                '%s: %s, see http://developer.bada.com/article/push-messaging-guide#status_codes_messages for details' %\
                    (response_in_json['results'][0]['statusCode'], response_in_json['results'][0]['statusMsg']))
        return request_serialized, response.content


def send(**kwargs):
    '''
    Shortcut for PushMessage(**kwargs).send()
    '''
    PushMessage(**kwargs).send()
