import logging
import inspect
import pprint

from pybox.utils import *

from abc import ABCMeta, abstractmethod

logger = logging.getLogger('pybox')

class Field(object):
    """
    A Field directive is used to define methods of extracting values from
    raw API objects. If a field is initialized with a string, it will use
    that string as a key to extract the the raw API object.
    """
    def __init__(self, field_or_fget):
        if type(field_or_fget) == str:

            def nget(raw_obj, keys):
                return reduce(lambda d,k: d[k], keys, raw_obj)

            self.fget       = lambda self, raw_obj: nget(raw_obj, field_or_fget.split('.'))
            self.field_name = field_or_fget.replace('.', '_')
        else:
            self.fget       = field_or_fget
            self.field_name = field_or_fget.__name__

        self.field_name = "__field_{}".format(self.field_name)

    def __get__(self, resource, resourcetype):
        return getattr(resource, self.field_name)

    def __set__(self, resource, val):
        setattr(resource, self.field_name, val)


class LazyField(Field):
    """
    A LazyField directive functions similarly to the Field directive. However,
    the extraction of a given field isn't required on initialization. Furthermore,
    if the value is available on initialization, the first access of the attribute
    will call the resources `reload()` method to make the expensive request to get
    the value.
    """
    def __get__(self, resource, resourcetype):
        # First try loading the lazy fields from the existing raw
        # object
        if not hasattr(resource, self.field_name):
            resource._reload_lazy_fields()

        # Then request new data from the server if the fields still
        # haven't been loaded.
        if not hasattr(resource, self.field_name):
            resource.reload()

        return getattr(resource, self.field_name)


class APIObject(object):
    """
    APIObject is a abstract class that is used to declaratively define mappings
    from raw json responses from an API, to native Python classes.
    """
    __metaclass__ = ABCMeta

    def __init__(self, raw_object):
        self.__raw_object = raw_object
        self._reload_fields()

        # Wrap reload function with _reload_fields() so that it will automatically
        # Extract the values.
        subclass_reload = self.reload
        def wrapped_reload():
            self.__raw_object = subclass_reload()

            logger.debug('Reloading {}. Raw Object:\n{}'
                         .format(self.__class__.__name__,
                                 pprint.pformat(self.__raw_object, width=1, indent=2)))
            self._reload_fields()
            self._reload_lazy_fields()

        self.reload = wrapped_reload

    def __repr__(self):
        return "<{} {}>".format(
            self.__class__.__name__,
            ', '.join(["%s=%s" % (k, getattr(self, k)) for k,_ in self.__fields]))

    @abstractmethod
    def reload(self):
        """
        Abstract method that should return a dict representing the
        json object returned by the API server. `_reload()` uses this value as
        the raw object from which it tries to extract the declared parameters.
        """
        raise Exception('Unimplemented method reload!')

    def _reload_fields(self):
        """
        Function responsible for extract all the fields from the raw api dictionary.
        All Fields must be extracted.
        """
        for field, v in self.__fields:
            try:
                setattr(self, field, v.fget(self, self.__raw_object))
            except:
                raise Exception(
                    'Unable to extract required field \'{}\'\n'.format(field) +
                    'from raw object: \n {}'.format(pprint.pformat(self.__raw_object,
                                                                   width=1,
                                                                   indent=2)))

    def _reload_lazy_fields(self):
        """
        Tries to extract the lazy field attributes from self.__raw_object
        It'll fail quitely if none of the lazy fields can be loaded.
        If some are loaded and some aren't, it'll raise and exception.
        """
        # If all LazyFields exist, set them, otherwise raise exception
        lazy_fields = [(field, suppress(lambda: v.fget(self, self.__raw_object)))
                        for field, v in self.__lazy_fields]
        lazy_errors = [type(v) == SuppressedError for k,v in lazy_fields]

        if all(lazy_errors):
            logger.debug('Unable to extract any lazy fields for {}. Assuming minispec.'
                         .format(self))
        elif any(lazy_errors):
            logger.debug('Raw object:\n{}'
                         .format(pprint.pformat(self.__raw_object, width=1, indent=2)))

            logger.debug(('Only able to extract some of the lazy fields for {}. ' +
                          'Disparity between local spec and server response.')
                          .format(self.__class__.__name__))
            logger.debug('Failed extracting {} for class: {}'
                         .format(', '.join(["'%s'" % field for field, v in lazy_fields
                                                           if type(v) == SuppressedError ]),
                                 self.__class__.__name__))
            # Raise one of the suppressed errors
            err = next(v for field, v in lazy_fields if type(v) == SuppressedError)
            raise err.type, err.value, err.traceback

        else:
            for field, v in lazy_fields:
                setattr(self, field, v)

    @property
    def __fields(self):
        return self.__attributes_of_type(Field)

    @property
    def __lazy_fields(self):
        return self.__attributes_of_type(LazyField)

    def __attributes_of_type(self, typ):
        """
        Get all attributes on self with the supplied type.
        Does so without calling `getattr()`
        """
        return [(attr, val) for cls in inspect.getmro(self.__class__)
                           for attr, val in cls.__dict__.iteritems()
                           if type(val) == typ]

