import logging
import os
import functools
import requests
from urllib import urlencode
from urlparse import urlparse

from pybox.utils import urlopen
from pybox.exceptions import *

logger = logging.getLogger('pybox')

def refreshable(func):
    @functools.wraps(func)
    def wrapped(session, *args, **kwargs):
        try:
            return func(session, *args, **kwargs)
        except BoxAPIException as e:
            if e.error == 'invalid_token':
                logger.info('Box Session expired. Attempting to refresh access token.')
                session.refresh_access_token()
                val = func(session, *args, **kwargs)
                logger.info('Box Session successfully refreshed.')
                return val
            else:
                raise

    return wrapped


BOX_BASE_URL  = 'https://app.box.com/api/'
BOX_TOKEN_URL = BOX_BASE_URL + 'oauth2/token'

class Session(object):
    """
    Box.net session object.
    This class manages Box.net credentials and provides methods
    to get bound urls.
    """
    def __init__(self, client_id, client_secret, access_token=None, refresh_token=None):
        self.client_id     = client_id
        self.client_secret = client_secret
        self.access_token  = access_token
        self.refresh_token = refresh_token

    def authorize_url(self, url, state=None):
        """
        :returns authorization_url: url user must visit to authorize specified application.
        """
        data = { 'client_id' : self.client_id,
                 'response_type' : 'code',
                 'redirect_uri' : url }

        if state:
            data['state'] = state

        return 'https://app.box.com/api/oauth2/authorize?%s' % urlencode(data)

    def refresh_access_token(self):
        """
        Using the supplied refresh token, we will try to refresh the
        session's access token.
        :param refresh_token: The refresh token used to refresh the access token.
        :returns refresh_token: The new refresh token that Box generated.
        """
        response = urlopen(requests.Request(url=BOX_TOKEN_URL, data={
           'client_id'     : self.client_id,
           'client_secret' : self.client_secret,
           'refresh_token' : self.refresh_token,
           'grant_type'    : 'refresh_token'
        }, method='POST'))
        logger.debug(response)
        self.access_token  = response['access_token']
        self.refresh_token = response['refresh_token']
        return (response['access_token'], response['refresh_token'])

    @refreshable
    def authorized_request(self, url, data=None, files={}, headers={},
                                      params={}, method='GET'):
        """
        Wrapper around requests to make authenticated requests to box.net
        """
        logger.debug('Making request to {} with params: {}'.format(url, params))
        # Prevent accidentally leaking credentials
        assert urlparse(url).netloc.endswith('box.com'),\
               'Can only make authorized requests to the Box API'

        if not self.access_token:
            raise Exception('No access token attached to this session.\
                             Must first authenticate.')

        headers = dict(self.authorization_header(), **headers)
        return urlopen(requests.Request(url=url, data=data, files=files,
                                        headers=headers, method=method,
                                        params=params))

    def authorization_header(self):
        return {'Authorization' : "Bearer {}".format(self.access_token)}


class CLISession(Session):
    """
    Session object that stores session credentials in ~/.pybox for future use.
    """
    def __init__(self, client_id, client_secret, access_token=None, refresh_token=None):
        self._credential_path = os.path.expanduser('~/.pybox')
        if os.path.isfile(self._credential_path):
            with open(self._credential_path, 'r') as f:
                access_token = access_token or f.readline()[:-1]
                refresh_token = refresh_token or f.readline()[:-1]

        super(CLISession, self).__init__(client_id, client_secret,
                                         access_token, refresh_token)
        self._write_credentials()

    def refresh_access_token(self, *args, **kwargs):
        super(CLISession, self).refresh_access_token(*args, **kwargs)
        self._write_credentials()

    def _write_credentials(self):
        with open(self._credential_path, 'w') as f:
            f.write(self.access_token+'\n')
            f.write(self.refresh_token+'\n')

