from pybox import api
from pybox.item import Item
from pybox.file import File
from pybox.metaclient import *
from pybox.utils import LazyList, LazyDict

class Folder(Item, LazyDict):
    item_count = LazyField('item_collection.total_count')

    def __init__(self, *args, **kwargs):
        Item.__init__(self, *args, **kwargs)

        # Wrap the item_collection in generator to avoid
        # eagerly making API requests to load the item collection
        def item_collection_iterator(self):
            for item in self.item_collection:
                yield (item.name, item)

        LazyDict.__init__(self, item_collection_iterator(self))

    def reload(self):
        return api.get_folder_by_id(self.session, self.id)

    @LazyField
    def parent(self, raw_object):
        parent = raw_object['parent']
        if parent and 'id' in parent:
            return Folder(self.session, api.get_folder_by_id(self.session, parent['id']))
        else:
            return None

    @LazyField
    def item_collection(self, raw_object):
        total_items = raw_object['item_collection']['total_count']
        items       = raw_object['item_collection']['entries']
        folder_id   = raw_object['id']

        def item_collection_iterator(session):
            def convert_item(session, item):
                if item['type'] == 'folder':
                    return Folder(session, item)
                elif item['type'] == 'file':
                    return File(session, item)

            for item in items:
                yield convert_item(session, item)

            for x in range(len(items), total_items, 1000):
                for item in api.get_items_for_folder_id(session, folder_id,
                                                        limit=1000, offset=x)['entries']:
                    yield convert_item(session, item)

        return LazyList(item_collection_iterator(self.session))

    @property
    def folders(self):
        return LazyDict((folder.name, folder) for folder in self if type(folder) == Folder)

    @property
    def files(self):
        return LazyDict((f.name, f) for f in self if type(f) == File)

    def upload(self, fp, filename):
        return File(self.session, api.upload_file_into_folder_by_id(
                                  self.session, fp,
                                  filename, self.id)['entries'][0])

    def create_subfolder(self, folder_name):
        return Folder(self.session, api.create_folder_in_folder_by_id(
                                    self.session, folder_name, self.id))

    def delete(self, recursive=False):
        return api.delete_folder_by_id(self.session, self.id,
                                       recursive=recursive,
                                       if_match_etag=self.etag)


