import sys
import requests
import logging
import urllib2
import json
from contextlib import contextmanager

from collections import deque

from pybox.exceptions import *

logger = logging.getLogger('pybox')

class LazyList(object):
    """ A List the is seeded lazily from an iterator """
    def __init__(self, iterator):
        self.data = []
        self._iterator = iter(iterator)

    def __getitem__(self, index):
        while len(self.data) <= index:
            self.data.append(self._iterator.next())
        return self.data[index]

class LazyDict(dict):
    """ A Collection that is based on an iterable list """
    def __init__(self, iterator):
        super(LazyDict, self).__init__()
        self._iterator = iter(iterator)

    def __iter__(self):
        for key, item in self._iterator:
            self.__setitem__(key, item)

        return super(LazyDict, self).itervalues()

    def __contains__(self, key):
        for k,v in self._iterator:
            self.__setitem__(k,v)

        return super(LazyDict, self).__contains__(key)

    def __getitem__(self, key):
        if key in self:
            return super(LazyDict, self).__getitem__(key)
        else:
            for k, item in self._iterator:
                self.__setitem__(k, item)
                if k == key:
                    return item
            raise IndexError()


def urlopen(request, timeout=30):
    session = requests.Session()
    response = session.send(request.prepare())
    session.close()

    if response.status_code == 204:
        logger.debug("No content")
        return ""

    if response.status_code == 200:
        if response.headers['Content-Type'] != 'application/json':
            return response.raw.read()
        else:
            return response.json()

    if response.status_code == 401:
        raise BoxAPIException('invalid_token',
                'The access token provided is expired, revoked, malformed or ' +
                'invalid for other reasons.')

    if response.status_code == 405:
        raise BoxAPIException('method_not_allowed',
                'Method not allowed.')

    response_obj = response.json()
    if 'error' in response_obj:
        raise BoxAPIException(response_obj['error'], response_obj['error_description'])
    elif 'code' in response_obj:
        raise BoxAPIException(response_obj['code'], response_obj['message'])

    return response_obj


class SuppressedError(object):
    def __init__(self):
        (self.type, self.value, self.traceback) = sys.exc_info()

def suppress(func):
    try:
        return func()
    except Exception as e:
        return SuppressedError()


@contextmanager
def file_or_path(f_or_path, mode='w'):
    # Open the file if given a path, otherwise pass the file descriptor on through
    fp = open(f_or_path, mode) if type(f_or_path) == str else f_or_path

    yield fp

    # If we opened a file, close it
    if type(f_or_path) == str:
        fp.close()



