import unittest
import urllib2
from mock import Mock, patch

from pybox.utils import *
from pybox.exceptions import *

class TestUtils(unittest.TestCase):

    def test_parse_no_content(self):
        resp = Mock(code=204)
        self.assertEquals(parse_response(resp), '')

    def test_parse_json(self):
        resp = Mock(code=200, headers={
                'Content-Type' : 'application/json'
            })
        resp.read.return_value = '{ "foo" : "bar" }'
        val = parse_response(resp)
        self.assertIn('foo', val)
        self.assertEquals(val['foo'], 'bar')

    def test_parse_nop(self):
        val = parse_response(None)
        self.assertIsNone(val)

    def test_parse_error(self):
        resp = Mock(code=200, headers={
                'Content-Type' : 'application/json'
            })
        resp.read.return_value = '{ "error" : "Fire!",' +\
                                 '  "error_description" : "OMG FIRE!" }'

        with self.assertRaises(BoxAPIException):
            val = parse_response(resp)

    def test_parse_not_json_is_identity(self):
        resp = Mock(code=200, headers={
                'Content-Type' : 'txt'
            })
        val = parse_response(resp)
        self.assertEquals(val, resp)

class TestLazyList(unittest.TestCase):
    def test_get_item(self):
        l = LazyList([1,2,3])
        self.assertEquals(l[0], 1)
        self.assertEquals(l[1], 2)
        self.assertEquals(l[2], 3)

    def test_get_item_generator(self):
        def gen():
            for x in range(1, 4):
                yield x

        l = LazyList(gen())
        self.assertEquals(l[0], 1)
        self.assertEquals(l[1], 2)
        self.assertEquals(l[2], 3)

    def test_is_iterable(self):
        def gen():
            for x in range(0, 3):
                yield x

        l = LazyList(gen())

        for index, x in enumerate(l):
            self.assertEquals(x, index)

        # Make sure you can page through more than once
        for index, x in enumerate(l):
            self.assertEquals(x, index)


class TestLazyDict(unittest.TestCase):

    def test_getitem(self):
        def gen():
            for index, x in enumerate('abcd'):
                yield x, index
        d = LazyDict(gen())

        self.assertEquals(d['a'], 0)
        self.assertEquals(d['b'], 1)
        self.assertEquals(d['c'], 2)
        self.assertEquals(d['d'], 3)

    def test_getitem_lazy(self):
        def gen():
            for index, x in enumerate('abcd'):
                yield x, index

        iterator = gen()

        d = LazyDict(iterator)

        self.assertEquals(d['a'], 0)
        self.assertEquals(d['c'], 2)
        self.assertEquals(('d', 3), iterator.next())
        self.assertEquals(d['b'], 1)

    def test_key_iterator(self):
        def gen():
            for index, x in enumerate('abcd'):
                yield x, index

        d = LazyDict(gen())

        for x in 'abcd':
            self.assertIn(x, list(d))

        self.assertEquals(len(list(d)), 4)

    def test_iteritems(self):
        def gen():
            for index, x in enumerate('abcd'):
                yield x, index

        d = LazyDict(gen())

        for k,v in d.iteritems():
            self.assertIn(k, 'abcd')
            self.assertIn(k, [0,1,2,3])


