# -*- coding: utf-8 -*-

class StrictDict(dict):
    """
    A simple :class:`dict` subclass. It prevents overriding existing keys and lazy deletion.
    """

    def __init__(self, seq=None, **kwargs):
        super().__init__(seq=seq, **kwargs)

    def __setitem__(self, key, value):
        """
        Sets a key/value pair. Raises a :class:`KeyError` if the key already exist.

        :param key: The key
        :type key: *
        :param value: The value
        :type value: *

        :raises: :exc:`KeyError`

        """
        if key in self:
            raise KeyError('Key "%s" is already taken. "StrictDict" does not allow key overrides.' % key)
        super().__setitem__(key, value)

    def __delitem__(self, key):
        """
        Removes a key/value pair based upon the given *key*. Raises a :class:`KeyError` if the key doesn't exist.

        :param key: The key
        :type key: *

        :raises: :exc:`KeyError`

        """
        if not key in self:
            raise KeyError('Key "%s" does not exist. "StrictDict" does not allow to lazily delete non-existing keys.' % key)
        super().__delitem__(key)