# -*- coding: utf-8 -*-
from inspect import signature, Parameter
import threading

from .annotators import Annotator, from_key, get_key
from .exceptions import AnnotationNotFound, BindingNotFound, InvalidAnnotation, DIError, InjectionFailed, UnknownParameterKind
from .types import StrictDict
from .providers import MainProviderProxy


class Injector(object):
    # TODO: Implement settings if required.
    """
    The ``Injector`` is responsible for binding and injecting dependencies.

    :arg binding_specs: A list of binding specs.
    :type binding_specs: list of :class:`~di.binding_specs.BindingSpec`.

    :keywords settings: A dictionary of settings.
    :type settings: dict

    """
    def __init__(self, *binding_specs, **settings):
        self._settings = settings
        self._bindings = StrictDict()
        self._cache = StrictDict()
        self._rlock = threading.RLock()
        with self._rlock:
            for spec in binding_specs:
                spec.configure(self.bind)

    def bind(self, name):
        """
        Starts a binding chain.

        :param name: The name of the binding
        :type name: str

        :rtype: :class:`~di.annotators.Annotator`
        """
        return Annotator(self, name, MainProviderProxy(self))

    def inject_into(self, item, *args, **kwargs):
        """
        Tries to inject dependencies into ``item``.

        :param item: The item you want to have dependencies injected to.
        :type item: type or :class:`types.FunctionType`

        :returns: A new instance of *(class)* or *from (function)* ``item``.
        :rtype: *
        """
        # TODO: Remove overloading in favor of a more concise way of skipping, substituting and so on.
        """
        Injects dependencies into a given ``item`` function or class constructor. The following parameters will get processed:

            - Arguments that can be passed as either positional or keyword (:class:`inspect.Parameter.POSITIONAL_OR_KEYWORD`)
            - Positional only arguments (:class:`inspect.Parameter.POSITIONAL_ONLY`)

        Additionally, overloaded positional and keyword arguments will get merged into the injection
        parameters (:class:`inspect.Parameter.VAR_POSITIONAL` and :class:`inspect.Parameter.VAR_POSITIONAL` respectively)

        .. note::
           Overloaded positionals (``*args``) and keywords (``**kwargs``) won't get passed to nested dependencies.

        :param item: A function or class
        :type item: type, :class:`types.FunctionType`

        :raises: :exc:`~di.exceptions.UnknownParameterKind` If the parameter is unknown
        :raises: :exc:`~di.exceptions.InjectionFailed`
        :rtype: *
        """

        # Determine whether to inspect the item itself or its constructor if present
        item_to_reflect = item.__init__ if isinstance(item, type) else item
        sig = signature(item_to_reflect)

        positionals = []
        keywords = {}

        for name, param in sig.parameters.items():
            # Skip 'self' and 'cls' parameters as they should never get injected.
            if name in ('self', 'cls', ):
                continue
            key = name
            # Extend key in case of an annotation present.
            if param.annotation is not Parameter.empty:
                if not isinstance(param.annotation, str):
                    raise InvalidAnnotation('"%r" is not a valid annotation to argument "%s" of "%r"' %
                        (param.annotation), name, item)
                key = get_key(name, str(param.annotation))
            # Collect positional parameters.
            if param.kind in (Parameter.POSITIONAL_OR_KEYWORD, Parameter.POSITIONAL_ONLY):
                try:
                    positionals.append(self.get_instance(key))
                except DIError as e:
                    raise InjectionFailed('Caught %s while injecting positional argument "%s" into "%r": "%s"'
                        % (e.__class__.__name__, name, item, str(e)))

            # Collect keyword only parameters.
            elif param.kind == Parameter.KEYWORD_ONLY:
                try:
                    keywords[name] = self.get_instance(key)
                except DIError as e:
                    raise InjectionFailed('Caught %s while injecting keyword parameter "%s" into "%r": "%s"'
                        % (e.__class__.__name__, name, item, str(e)))
            # Collect overloaded positionals and keywords.
            elif param.kind == Parameter.VAR_POSITIONAL:
                positionals += args
            elif param.kind == Parameter.VAR_KEYWORD:
                passed_kwargs = kwargs
                passed_kwargs.update(keywords)
                keywords = passed_kwargs
            else:
                raise UnknownParameterKind('Unexpected parameter kind %s' % param.kind)

        # Sanitize parameters. Use ``bind_partial`` because a complete ``bind`` would fail in case of
        # a signature containing instance (``'self'``) or class (``cls``) arguments.
        bound = sig.bind_partial(*positionals, **keywords)
        return item(*bound.args, **bound.kwargs)

    def provide(self, item, *args, **kwargs):
        """
        Shortcut. Never used by internals. Based on the ``item`` type does one of the following:

            - If ``isinstance(item, str)`` delegate to :meth:`~di.contexts.Injector.get_instance`
            - Else delegate to :meth:`~di.contexts.Injector.inject_into`

        :param item: A value or binding name that should get provided.
        :type item: str or type or :class:`types.FunctionType`

        :returns: A new instance of *(class)* or *from (function)* ``item`` .
        :rtype: *
        """
        if isinstance(item, str):
            return self.get_instance(item, *args, **kwargs)
        return self.inject_into(item, *args, **kwargs)

    def get_instance(self, key, *args, **kwargs):
        """
        Returns item provided by the provider that is bound to ``key``.

        :param key: The binding key.
        :type key: str

        :raises: :exc:`~di.exceptions.BindingNotFound`
        :raises: :exc:`~di.exceptions.AnnotationNotFound`

        :rtype: *
        """
        try:
            return self._bindings[key].provide(*args, **kwargs)
        except KeyError:
            name, annotation = from_key(key)
            if name in self._bindings:
                raise AnnotationNotFound('Binding "%s" annotated with "%s" could not be found.'
                    % (name, annotation))
            else:
                raise BindingNotFound('Binding  "%s" could not be found.' % key)


