# -*- coding: utf-8 -*-
from inspect import signature, Parameter
from unittest import TestCase
from di.injectors import Injector


class Foo(object):
    def __init__(self, name, status, *args, profession, **kwargs):
        self.name = name
        self.status = status
        self.args = args
        self.profession = profession
        self.kwargs = kwargs


class FooWrapper(object):
    def __init__(self, foo):
        self.foo = foo

class AnotherFooWrapper(object):
    def __init__(self, another_foo):
        self.another_foo = another_foo

class Bar(object):
    def __init__(self, status, *args, profession, **kwargs):
        self.status = status
        self.profession = profession
        self.args = args
        self.kwargs = kwargs
        self.name = 'Generic jedi'


class BarWrapper(object):
    def __init__(self, bar):
        self.foo = bar


def foo_factory(*args, profession:'dark', **kwargs):
    return Foo('Anakin', 'Dead', profession=profession, *args, **kwargs)


def foo_factory_wrapper(foo_factory):
    return foo_factory


def bar_factory(*args, profession:'dark', **kwargs):
    return Bar('Alive', profession=profession, *args, **kwargs)


def bar_factory_wrapper(bar_factory):
    return bar_factory


class TestInspectPossibilities(TestCase):

    def setUp(self):
        self.s = signature(Foo.__init__)

    def test_signature(self):
        self.assertIn('self', self.s.parameters)

    def test_params(self):

        args = []
        kwargs = {}
        for name, param in self.s.parameters.items():
            if name in ('self', ):
                continue
            if param.kind in (Parameter.POSITIONAL_OR_KEYWORD, Parameter.POSITIONAL_ONLY):
                args.append('%s as positional' % name)
            elif param.kind == Parameter.KEYWORD_ONLY:
                kwargs[name] = '%s as keyword' % name
            elif param.kind == Parameter.VAR_POSITIONAL:
                args += ['some', 'overloaded', 'args']
            elif param.kind == Parameter.VAR_KEYWORD:
                passed_kwargs = {
                    'a': 'some',
                    'b': 'overloaded',
                    'c': 'kwargs',
                }
                passed_kwargs.update(kwargs)
                kwargs = passed_kwargs
            else:
                raise Exception('Unexpected Parameter kind %s' % param.kind)

        bound_args = self.s.bind(*args, **kwargs)
        foo = Foo(*bound_args.args, **bound_args.kwargs)

        self.assertEqual(foo.name, 'name as positional')
        self.assertEqual(foo.status, 'status as positional')
        self.assertEqual(foo.args, ('some', 'overloaded', 'args', ))
        self.assertEqual(foo.profession, 'profession as keyword')
        self.assertEqual(foo.kwargs, {
            'a': 'some',
            'b': 'overloaded',
            'c': 'kwargs'
        })


class TestProvider(TestCase):

    def setUp(self):
        self.graph = Injector()
        self.graph.bind('name').to_value('Luke')
        self.graph.bind('status').to_value('Alive')
        self.graph.bind('profession').to_value('Jedi')
        self.graph.bind('profession').annotated_with('dark').to_value('Sith')
        self.graph.bind('foo').to_class(Foo).as_singleton()
        self.graph.bind('another_foo').to_class(Foo).as_singleton()
        self.graph.bind('bar').to_class(Bar).as_prototype()
        self.graph.bind('foo_factory').to_function(foo_factory).as_singleton()
        self.graph.bind('bar_factory').to_function(bar_factory).as_prototype()

    def test_binding_is_set(self):
        self.assertIn('name', self.graph._bindings)

    def test_value_provider(self):
        self.assertEqual(self.graph.provide('status'), 'Alive')

    def test_annotated_value_provider(self):
        self.assertEqual(self.graph.provide('profession'), 'Jedi')
        self.assertEqual(self.graph.provide('profession:dark'), 'Sith')

    def test_constructor_provider(self):
        foo = self.graph.provide(Foo)
        self.assertEqual(foo.name, 'Luke')
        self.assertEqual(foo.status, 'Alive')
        self.assertEqual(foo.profession, 'Jedi')
        self.assertEqual(foo.args, ())
        self.assertEqual(foo.kwargs, {})

    def test_function_provider(self):
        foo = self.graph.provide(foo_factory)
        self.assertEqual(foo.name, 'Anakin')
        self.assertEqual(foo.status, 'Dead')
        self.assertEqual(foo.profession, 'Sith')
        self.assertEqual(foo.args, ())
        self.assertEqual(foo.kwargs, {})

    def test_nested_provider(self):
        wrapper = self.graph.provide(FooWrapper)
        self.assertEqual(wrapper.foo.name, 'Luke')
        self.assertEqual(wrapper.foo.status, 'Alive')
        self.assertEqual(wrapper.foo.profession, 'Jedi')
        self.assertEqual(wrapper.foo.args, ())
        self.assertEqual(wrapper.foo.kwargs, {})

    def test_singleton_behaviour(self):
        wrapper = self.graph.provide(FooWrapper)
        another_wrapper = self.graph.provide(FooWrapper)
        self.assertEqual(wrapper.foo, another_wrapper.foo)

    def test_prototype_behaviour(self):
        wrapper = self.graph.provide(BarWrapper)
        another_wrapper = self.graph.provide(BarWrapper)
        self.assertNotEqual(wrapper.foo, another_wrapper.foo)

    def test_function_singleton_behaviour(self):
        wrapper = self.graph.provide(foo_factory_wrapper)
        another_wrapper = self.graph.provide(foo_factory_wrapper)
        self.assertEqual(wrapper, another_wrapper)

    def test_function_prototype_behaviour(self):
        wrapper = self.graph.provide(bar_factory_wrapper)
        another_wrapper = self.graph.provide(bar_factory_wrapper)
        self.assertNotEqual(wrapper, another_wrapper)

    def test_singleton_cache(self):
        foo_wrapper = self.graph.provide(FooWrapper)
        another_foo_wrapper = self.graph.provide(AnotherFooWrapper)
        self.assertEqual(foo_wrapper.foo, another_foo_wrapper.another_foo)