# -*- coding: utf-8 -*-
import threading

ANNOTATION_FORMAT = '{name}:{annotation}'
"""Standard binding key format for name/annotation pairs """


def get_key(name, annotation=None):
    """
    Returns the binding key for a name/annotation pair, formatted with :attr:`~di.annotators.ANNOTATION_FORMAT`,
    e.g. ``'widget:text'``.

    :param name: A binding name
    :type name: str

    :param annotation: A binding annotation
    :type annotation: str

    :rtype: str
    """
    if annotation:
        return ANNOTATION_FORMAT.format(**dict(name=name, annotation=annotation))
    return name


def from_key(key):
    """
    Returns a tuple ``(name, annotation)`` for a key formatted as defined in
    :attr:`~di.annotators.ANNOTATION_FORMAT`, e.g. ``('widget', 'text')`` for key ``'widget:text'``.

    :param key: A key
    :type key: str

    :rtype: tuple
    """
    parts = key.split(':', 1)
    return (parts[0], None) if len(parts) < 2 else parts


class Annotator(object):
    """
    Handles binding annotations by providing an extended binding key to the passed ``injector`` in case of an annotation
    request. If not, delegating attributes to ``provider``. In both cases the actual ``key -> value`` assignment
    happens within the ``Annotator``.

    :param injector: The injector that hosts bindings.
    :type injector: :class:`~di.injectors.Injector`

    :param name: The name of the binding that eventually will get annotated.
    :type name: str

    :param provider: A provider object that you want to bind to a name/annotation pair.
    :type provider: :class:`~di.providers.Provider`
    """
    def __init__(self, injector, name, provider):
        self._injector = injector
        self._name = name
        self._provider = provider
        self._rlock = threading.RLock()

    def annotated_with(self, annotation):
        """
        Extends the binding key with ``annotation`` and delegates further provider instructions to (returns)
        the wrapped provider. Adds

        :param annotation: An annotation name.
        :type annotation: str

        :rtype: :class:`~di.providers.Provider`
        """
        with self._rlock:
            key = get_key(self._name, annotation)
            self._injector._bindings[key] = self._provider
            return self._provider

    def __getattr__(self, item):
        """
        Tries to delegate attribute calls to the wrapped provider.

        :rtype: :class:`~di.providers.Provider`
        """
        with self._rlock:
            attr = getattr(self._provider, item, None)
            if attr is None:
                raise AttributeError('Neither "Annotator" nor "%s" has ')
            else:
                self._injector._bindings[get_key(self._name)] = self._provider
                return attr





