# -*- coding: utf-8 -*-
import threading
import types
from .exceptions import NoInnerProviderFound, InvalidBindingItem

class Provider(object):
    """
    Basic interface for providers. Providers (or chain of providers) defines the way a binding gets injected upon
    request.

    :param injector: The injector hosting the DI.
    :type injector: :class:`~di.injectors.Injector`

    :param item: The item you want to provide.
    :type item: *
    """
    def __init__(self, injector, item=None):
        self._injector = injector
        self._item = item
        self._rlock = threading.RLock()

    def provide(self, *args, **kwargs):
        """
        Provides an object according to configuration and implementation.

        :raises: NotImplemented
        :rtype: *
        """
        raise NotImplemented


class ProviderProxy(Provider):
    """
    A proxy class that delegates provide request to an inner provider object. The ``ProviderProxy`` object should
    expose an interface to set its inner providers.
    """

    _provider = None

    def provide(self, *args, **kwargs):
        """
        Delegates the call to its inner provider.

        :raises: :exc:`~di.exceptions.ProviderProxyError`
        :rtype: *
        """
        with self._rlock:
            if self._provider is None:
                raise NoInnerProviderFound('"%s" can\'t delegate to inner provider because it has none set.'
                                         % self.__class__.__name__)
            return self._provider.provide(*args, **kwargs)


class SingletonProvider(Provider):
    """
    A singleton provider. The item gets provided and when first requested and then cached for further requests.
    """
    _instance = None

    def __init__(self, injector, item=None):
        super().__init__(injector, item)

    def provide(self, *args, **kwargs):
        """
        Please note that no ``args`` or ``kwargs`` will get passed to the injection process, as it would be a huge
        liability to create a singleton with eventual runtime variables.

        :rtype: *
        """
        with self._rlock:
            if not self._item in self._injector._cache:
                self._injector._cache[self._item] = self._injector.inject_into(self._item)
            return self._injector._cache[self._item]


class PrototypeProvider(Provider):
    """
    A prototype provider. Returns a newly provided ``item`` for each request.
    """
    def provide(self, *args, **kwargs):
        """
        Returns a newly created object.

        :rtype: *
        """
        with self._rlock:
            return self._injector.inject_into(self._item, *args, **kwargs)


class ValueProvider(Provider):
    """
    A value provider. Returns a static value.
    """
    def __init__(self, injector, item=None):
        super().__init__(injector, item)
        self._rlock = threading.RLock()

    def provide(self, *args, **kwargs):
        with self._rlock:
            return self._item


class ScopeProxy(ProviderProxy):
    """
    A proxy exposing an interface for scope selection.
    """
    def as_singleton(self):
        """
        Sets a :class:`~di.providers.SingletonProvider` as inner provider.
        """
        with self._rlock:
            self._provider = SingletonProvider(self._injector, self._item)
            return self._provider

    def as_prototype(self):
        """
        Sets a :class:`~di.providers.PrototypeProvider` as inner provider.
        """
        with self._rlock:
            self._provider = PrototypeProvider(self._injector, self._item)
            return self._provider


class FunctionScopeProxy(ScopeProxy):
    """
    A proxy exposing an interface for scope selection when binding function.

    .. note::
       At the moment, :class:`~di.providers.FunctionScopeProxy` is just a plain subclass of
       :class:`~di.providers.ScopeProxy` with no specific implementations whatsoever. It exists mostly for
       semantic reasons and with eventual future changes in mind.
    """
    pass


class ClassScopeProxy(ScopeProxy):
    """
    A proxy exposing an interface for scope selection when binding classes (types).

    .. note::
       At the moment, :class:`~di.providers.ClassScopeProxy` is just a plain subclass of
       :class:`~di.providers.ScopeProxy` with no specific implementations whatsoever. It exists mostly for
       semantic reasons and with eventual future changes in mind.
    """
    pass


class MainProviderProxy(ProviderProxy):
    """
    The main entry point for a binding chain. Exposes factory methods to determine the ``item`` type provider.
    """
    def to_value(self, item):
        """
        Sets a :class:`~di.providers.ValueProvider` object as inner provider.

        :param item: The item you want to provide for the current binding chain.
        :type item: *

        :rtype: :class:`~di.providers.ValueProvider`
        """
        with self._rlock:
            self._provider = ValueProvider(self._injector, item)
            return self._provider

    def to_class(self, item):
        """
        Sets a :class:`~di.providers.ClassScopeProxy` object as inner provider that later on should
        determine the scope the provider should work in.

        :param item: The item you want to provide for the current binding chain.
        :type item: type

        :rtype: :class:`~di.providers.ClassScopeProxy`
        """
        with self._rlock:
            if not isinstance(item, type):
                raise InvalidBindingItem('"%r" is not a class.')
            self._provider = ClassScopeProxy(self._injector, item)
            return self._provider

    def to_function(self, item):
        """
        Sets a :class:`~di.providers.FunctionScopeProxy` object as inner provider that later on should
        determine the scope the provider should work in.

        :param item: The item you want to provide for the current binding chain.
        :type item: :class:`types.FunctionType`

        :rtype: :class:`~di.providers.FunctionScopeProxy`
        """
        with self._rlock:
            if not isinstance(item, types.FunctionType):
                raise InvalidBindingItem('"%r" is not a function.')
            self._provider = FunctionScopeProxy(self._injector, item)
            return self._provider