#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright 2013 by Björn Johansson.  All rights reserved.
# This code is part of the Python-dna distribution and governed by its
# license.  Please see the LICENSE.txt file that should have been included
# as part of this package.

import tempfile
import sys
import os
import subprocess
import itertools
import operator
import shutil
from Bio           import SeqIO
from Bio.SeqRecord import SeqRecord
import dsdna

class Ape:

    def __init__(self, path_to_ape, tmpdir=None):
        '''
       >>> 1+1
       2
       '''
        path    = (path_to_ape,
                   path_to_ape.split("tcl")[1].strip(),
                   path_to_ape.split("tclsh")[1].strip(),)

        if True in [os.path.isfile(p) for p in path]:
            self.path_to_ape = path_to_ape
        else:
            print
            print path_to_ape
            print "is not a valid path to ApE"
            raise(ValueError("invalid path to ApE"))

        self.tmpdir = tmpdir or os.path.join(tempfile.gettempdir(),"ApE")
        try:
            os.makedirs(self.tmpdir)
        except OSError:
            pass

    def open(self,*args,**kwargs):
        '''
       >>> 5+5
       10
       '''
        args=list(args)
        for i, arg in enumerate(args):
            if not hasattr(arg, "__iter__") or isinstance(arg, SeqRecord):
                args[i] = (arg,)
        seqs = []
        names = []
        for arg in itertools.chain.from_iterable(args):
            seq=dsdna.drecord(arg)
            for feature in seq.features:
                qf = feature.qualifiers
                if not "label" in qf:
                    try:
                        qf["label"] = qf["note"]
                    except KeyError:
                        qf["label"] = "feat{}".format(len(feature))
                if not "ApEinfo_fwdcolor" in qf:
                    qf["ApEinfo_fwdcolor"]="cyan"
                if not "ApEinfo_revcolor" in qf:
                    qf["ApEinfo_revcolor"]="red"
            seq.features.sort(key = operator.attrgetter("location.start"))
            seqs.append(seq)
            name=seq.name
            n=1
            while True:
                if name in names:
                    newname=name+"_"+str(n)
                    if newname in names:
                        n+=1
                        continue
                    else:
                        names.append(newname)
                        break
                else:
                    names.append(seq.description)
                    break
        pathstofiles = []
        path = tempfile.mkdtemp(dir=self.tmpdir)

        for name, seq in zip(names,seqs):
            whole_path = "{}.gb".format(os.path.join(path, name))
            seq.write(whole_path)
            pathstofiles.append('"{}"'.format(whole_path))

        #print "{} {}".format(self.path_to_ape,
        #                       " ".join(pathstofiles))

        p = subprocess.Popen("{} {}".format(self.path_to_ape," ".join(pathstofiles)),
                             shell=True,
                             stdout = tempfile.TemporaryFile(),
                             stderr = tempfile.TemporaryFile()).pid

        #shutil.rmtree(path)

if __name__=="__main__":
    from Bio import SeqIO
    sr1 = SeqIO.parse("../tests/pUC19.gb","gb").next()
    sr2 = SeqIO.parse("../tests/pCAPs.gb","gb").next()
    aperunner = Ape("tclsh /home/bjorn/ApE/apeextractor/ApE.vfs/lib/app-AppMain/AppMain.tcl")
    aperunner.open(sr1,sr2)
