""" Items API

Echo Items API is designed for providing interface for working with items in Activity Streams format.
Items API is the core component of the Platform, which allows you to submit items and after that build
the various data sets using the Echo Query Language.
"""
from echo.core_api import EchoClient, StreamServerError
import mux_api

import logging

# ======================================================================

class Client(EchoClient):
    def submit(self, content, mode='update'):
        """ Submit items "content" in Activity Streams XML format. """
        param_d = {
            'content': content,
            'mode': mode,
        }
        return self._send_request('submit', param_d, http_post=True)

    def _query(self, query, api_method='search', since=None):
        """ EQL query is either "search" or "count".
        The JSON response will vary, but the parameters are the same.
        """
        param_d = {
            'appkey': self._account._appkey,
            'q': unicode(query),
        }
        # since: an optional, opaque timestamp value from a previous query.
        if since and api_method=='search': param_d['since'] = since
        return self._send_request(api_method, param_d, http_post=False)

    def search(self, query, since=None):
        """ Search for items using the Echo Query Language.

        The query is a valid EQL string like the one here.
        query="scope:http://www.example.com/*"

        since is an optional, opaque timestamp value from a previous query.
        """
        return self._query(query, api_method='search', since=since)

    def count(self, query):
        """ Count the number of items within a dataset generated by a certain query, using Echo Query Language.

        The query is a valid EQL string like the one here.
        query="scope:http://www.example.com/*"
        """
        r = self._query(query, api_method='count', since=None)
        if 'count' in r:
            return r['count']
        else:
            return 0

    def mux(self, mux_requests):
        """ This single API call can "multiplex" search and count requests.
        mux_requests a list of echo.items.mux_api.MuxRequest objects.
        """
        requests = mux_api.requests_json(mux_requests)
        # TODO: limit requests to mux_api.MUX_MAX_REQUESTS
        param_d = {
            'appkey': self._account._appkey,
            'requests': requests,
        }
        return self._send_request('mux', param_d, http_post=True)

# Module-level client exposes API
# ======================================================================

def submit(content, mode='update'):
    """ Submit items "content" in Activity Streams XML format. """
    return Client.get_client().submit(content, mode=mode)

def search(query, since=None):
    """ Search for items using the Echo Query Language.

    The query is a valid EQL string like the one here.
    query="scope:http://www.example.com/*"

    since is an optional, opaque timestamp value from a previous query.
    """
    return Client.get_client().search(query, since=since)

def count(query):
    """ Count the number of items within a dataset generated by a certain query, using Echo Query Language.

    The query is a valid EQL string like the one here.
    query="scope:http://www.example.com/*"
    """
    return Client.get_client().count(query)

def mux(mux_requests):
    """ This single API call can "multiplex" search and count requests.
    mux_requests a list of echo.items.mux_api.MuxRequest objects.
    """
    return Client.get_client().mux(mux_requests)

