# -*- coding: utf-8 -*-
import logging
import re

from errors import TaskNotFound


class TasksManager(object):
    """
        | Make key from function path
        | And store it as pointer to functions which decorated @Pake.add_task
    """
    def __init__(self):
        """
            Defined private parameters
        """
        self._log = logging.getLogger("TasksManager")
        self._tasks = {}
        self._sorted_tasks = []

    def _make_prefix(self, path):
        """
            Make prefix from file path
        """
        if re.match(r"^tasks.pyc?$", path):
            prefix = ""
        else:
            prefix = re.sub(r"^\.?/?tasks/|\.pyc?$", "", path).replace("/", ":")
        return prefix

    def add_task(self, proc):
        """
            | Store task in internal dictionary
            | Store task doc string for overview
        """
        proc_fpath = proc.__globals__["__file__"]
        prefix = self._make_prefix(proc_fpath)
        key = (prefix and prefix + ":") + proc.__name__
        self._tasks[key] = proc
        doc = proc.__doc__ and proc.__doc__.strip()
        self._sorted_tasks.append((key, doc))

    def run(self, key, *args, **kwargs):
        """
            | Resolve alias if alias specify
            | Find task in internal dictionary
            | If not found raise :py:exc:`pake.runner.errors.TaskNotFound`
        """
        if key[0] == ":":
            alias = key
            key = self._sorted_tasks[int(alias[1:])][0]
            self._log.info("Replace \"{}\" => \"{}\"".format(alias, key))

        if key in self._tasks:
            return self._tasks[key](*args, **kwargs)
        else:
            raise TaskNotFound(key)

    def overview(self):
        """
            | Prepare report with tasks, represent as table Alias\|Task\|Comments
            | Which pake find:
            |   - $PWD/tasks.py
            |   - $PWD/tasks/\*.py
        """
        overview = None
        aliases, tasks, comments = ["Alias"], ["Task"], [["Comment"]]
        for i, (key, doc) in enumerate(self._sorted_tasks):
            aliases.append(u":{}".format(i))
            tasks.append(key)
            comments.append([line.strip() for line in unicode(doc).split(u"\n")])
        maxalias, maxtask = (
            len(max(aliases, key=len)),
            len(max(tasks, key=len))
        )
        comment_rjust = maxalias + maxtask + 4
        result = []
        for als, tsk, cmt in zip(aliases, tasks, comments):
            als = als.ljust(maxalias)
            tsk = tsk.ljust(maxtask)
            dump = []
            for i, line in enumerate(cmt):
                if i == 0:
                    dump.append(line)
                else:
                    dump.append(line.rjust(comment_rjust + len(line)))
            cmt = "\n".join(dump)
            result.append(u"{}  {}  {}".format(als, tsk, cmt))
        if len(result) > 1:
            overview = u"\n".join(result) + u"\n"
        return overview
