# -*- coding: utf-8 -*-
import shlex
import subprocess
import logging

from sudo import Sudo

log = logging.getLogger("pake.shell")

def debug_output(res):
    """
        | If output more than 30 lines
        | Split command debug output
        | Put first 15 lines and last 15 lines
    """
    split = res.split("\n")
    if len(split) > 30:
        res = split[:15] + [u"..."] + split[-15:]
        res = "\n".join(res)
    if res:
        log.debug("Out:\n{}".format(res))


class ShellPipe(object):
    def __init__(self, cmd):
        self._prev_cmds = []
        self._cmd = subprocess.Popen(cmd, stdout=subprocess.PIPE)
        self._report = [" ".join(cmd)]

    def pipe(self, cmd):
        """
            Add command to pipe
        """
        prev = self._cmd
        self._prev_cmds.append(prev)
        self._cmd = subprocess.Popen(cmd, stdin=prev.stdout, stdout=subprocess.PIPE)
        prev.stdout.close()
        self._report.append(" ".join(cmd))
        return self

    def last(self, cmd):
        """
            | log.info full pipe
            | Return pipe result
        """
        self.pipe(cmd)
        log.info("$ " + " | ".join(self._report))
        result = self._cmd.communicate()[0]
        [c.wait() for c in self._prev_cmds]
        debug_output(result)
        return result

    def __getattr__(self, attr):
        return ShellCmd(attr, pipe=self)


class ShellCmd(object):
    """
        Constructor for shell commands
    """
    def __init__(self, name, pipe=None):
        self._name = name
        self._pipe = pipe

    def _call(self, cmd, pipe):
        if pipe:
            return ShellPipe(cmd)
        else:
            log.info("$ " + " ".join(cmd))
            res = subprocess.check_output(cmd)
            debug_output(res)
            return res

    def _call_in_pipe(self, cmd, pipe):
        if pipe:
            return self._pipe.pipe(cmd)
        else:
            return self._pipe.last(cmd)

    def __call__(self, args="", pipe=False):
        cmd = u"{} {}".format(self._name, args)
        cmd = shlex.split(cmd)
        if not self._pipe:
            cmd = Sudo() + cmd
        if self._pipe:
            return self._call_in_pipe(cmd, pipe)
        else:
            return self._call(cmd, pipe)
