import ConfigParser
import json
from yaml import YAMLError
import yaml
from simplesettings.exceptions import SettingsException


class BaseParser(object):
    def __init__(self, filename):
        self.error_msg = 'Failed to %s settings file: %s.\nError was: %s'
        self.filename = filename
        self.content = self.read_settings_file()

    def read_settings_file(self):
        try:
            with open(self.filename) as fh:
                return fh.read()
        except (IOError, OSError) as e:
            raise SettingsException(self.error_msg % ('read', self.filename, e))


class JsonSettingsParser(BaseParser):
    def parse(self):
        try:
            output = json.loads(self.content)
        except ValueError as e:
            raise SettingsException(self.error_msg % ('parse', self.filename, e))
        return output


class YamlSettingsParser(BaseParser):
    def parse(self):
        try:
            output = yaml.load(self.content)
        except YAMLError as e:
            raise SettingsException(self.error_msg % ('parse', self.filename, e))
        return output


class IniSettingsParser(BaseParser):
    def parse(self):
        p = FileParser(allow_no_value=True)
        try:
            p.read(self.filename)
            output = p.as_dict()
        except ConfigParser.Error as e:
            raise SettingsException(self.error_msg % ('parse', self.filename, e))

        if output == []:
            raise SettingsException(self.error_msg % ('read', self.filename, 'Not found'))
        return output


class FileParser(ConfigParser.ConfigParser):
    def as_dict(self):
        d = dict(self._sections)
        for k in d:
            d[k] = dict(self._defaults, **d[k])
            d[k].pop('__name__', None)
        return d
