import types

from simplesettings.exceptions import SettingsException
from simplesettings.parsers import JsonSettingsParser, YamlSettingsParser, IniSettingsParser
from simplesettings.logger import logger
from simplesettings.vars import SimpleVars


OBJECT_ROUTER = {
    "yaml": YamlSettingsParser,
    "json": JsonSettingsParser,
    "ini": IniSettingsParser,
}


class Wrap(dict):
    """ Wrapper Class

        This wrapper class is used to separate settings from vars from the settings object and to split out
        dictionaries into a object.string.value model.
        We inherit from a dict, so all settings are both accessible as some['bla'] as well as some.bla

    """
    pass


class Settings(object):
    """
        Settings Object - The representation of a settings file

        Input:

         Filename   - The name of the settings file
         Parser     - The parser type, for now either json or yaml

         Output:

         A settings object with both data as a dictionary as well as split up into strings to prevent deep nesting.

    """
    mandatory_fields = []

    def __init__(self, filename, parser, mandatory_fields=()):
        try:
            file_parser = OBJECT_ROUTER[parser]
            self.data = Wrap()
            self.data.parser = file_parser(filename=filename)
            self.data.dict = self.data.parser.parse()
            self.vars = SimpleVars(input_dict=self.data.dict)
        except (TypeError, KeyError):
            raise SettingsException('File parser not found! Please use "ini, "json" or "yaml"')

        if self.mandatory_fields:
            mandatory_fields = self.mandatory_fields
        self.sanity(fields=mandatory_fields)

        for key in self.data.dict:
            if key in self.vars:
                continue

            value = self.data.dict[key]
            if isinstance(value, types.DictType):
                obj = Wrap()
                obj.update(value)
                obj.data_dict = value
                for element in value:
                    setattr(obj, element, value[element])
                setattr(self, key, obj)
            else:
                setattr(self, key, value)

        if self.data.dict.get('enable_logging', False):
            self.log = logger(
                name=self.vars.log_name,
                logfile=self.vars.log_file,
                verbosity=self.vars.verbosity)

    def sanity(self, fields):
        missing = [f for f in fields if f not in self.data.dict]
        if missing:
            raise SettingsException('Not all mandatory fields present! Missing: %s' % ','.join(missing))
