import types

from simplesettings.exceptions import SettingsException
from simplesettings.parsers import JsonSettingsParser, YamlSettingsParser, IniSettingsParser
from simplesettings.logger import logger
from simplesettings.vars import SettingVars


OBJECT_ROUTER = {
    "yaml": YamlSettingsParser,
    "json": JsonSettingsParser,
    "ini": IniSettingsParser,
}


class Wrap(dict):
    """ Wrapper Class

        This wrapper class is used to separate settings from vars from the settings object and to split out
        dictionaries into a object.string.value model.
        We inherit from a dict, so all settings are both accessible as some['bla'] as well as some.bla

    """
    pass


class Settings(object):
    """
        Settings Object - The representation of a settings file

        Input:

        Filename           - The name of the settings file

        Parser             - The parser type, for now either json or yaml

        Additional fields  - A dict with additional settings that should be included in the object

        Mandatory fields   - A list with keys that should be present in the settings or additional fields.
                             If not present, a SettingsException is raised.

         Output:

         A settings object with both data as a dictionary as well as split up into strings to prevent deep nesting.

    """
    def __init__(self, filename, parser, additional_fields={}, mandatory_fields=[]):
        try:
            file_parser = OBJECT_ROUTER[parser]
            self.data = Wrap()
            self.data.parser = file_parser(filename=filename)
            self.data.dict = self.data.parser.parse()
            self.data.dict.update(additional_fields)
            self.vars = SettingVars(input_dict=self.data.dict)
        except (TypeError, KeyError, ValueError):
            raise SettingsException('File parser not found! Please use "ini, "json" or "yaml"')

        self.sanity(fields=mandatory_fields)

        for key in self.data.dict:
            if key in self.vars:
                continue

            value = self.data.dict[key]
            if isinstance(value, types.DictType):
                obj = Wrap()
                obj.update(value)
                obj.data_dict = value
                for element in value:
                    setattr(obj, element, value[element])
                setattr(self, key, obj)
            else:
                setattr(self, key, value)

        if self.data.dict.get('enable_logging', False):
            self.log = logger(
                name=self.vars.log_name,
                logfile=self.vars.log_file,
                verbosity=self.vars.verbosity)

    def present(self, field):
        if self.data.dict.get(field, False) is False:
            return False
        return True

    def get(self, field):
        return self.data.dict.get(field, None)

    def sanity(self, fields):
        missing = [f for f in fields if f not in self.data.dict]
        if missing:
            raise SettingsException('Not all mandatory fields present! Missing: %s' % ','.join(missing))
        return True