"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
#
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""
import ast
import platform
import time
import datetime
import logging
from logging.config import fileConfig
import os
from threading import Lock
import traceback
import sys

from gevent import monkey
import gevent


logger = logging.getLogger("SolBase")
lifecyclelogger = logging.getLogger("LifeCycle")


class SolBase(object):
    """
    Base utilities & helpers.
    """

    #===============================
    # STATIC STUFF
    #===============================

    # Component name (mainly for rsyslog)
    _compoName = "CompoNotSet"

    # Global init stuff
    _voodooInitialized = False
    _voodooLock = Lock()

    # Logging stuff
    _loggingInitialized = False
    _loggingLock = Lock()

    # Fork stuff
    _masterProcess = True



    #===============================
    # DATE & MS
    #===============================

    @classmethod
    def msCurrent(cls):
        """
        Return current millis since epoc
        :return int
        :rtype int
        """
        return time.time() * 1000.0

    @classmethod
    def msDiff(cls, msStart, msEnd=None):
        """
        Get difference in millis between current millis and provided millis.
        :param msStart: Start millis
        :type msStart: float
        :param msEnd: End millis (will use current if not provided)
        :type msEnd: float
        :return float
        :rtype float
        """

        if msEnd:
            return msEnd - msStart
        else:
            return cls.msCurrent() - msStart

    @classmethod
    def dateCurrent(cls, eraseMode=0):
        """
        Return current date (UTC)
        :param eraseMode: Erase mode (0=nothing, 1=remove microseconds but keep millis, 2=remove millis completely)
        :return datetime
        :rtype datetime
        """

        if eraseMode == 0:
            return datetime.datetime.utcnow()
        elif eraseMode == 1:
            # Force precision loss (keep millis, kick micro)
            dt = datetime.datetime.utcnow()
            return dt.replace(microsecond=(dt.microsecond * 0.001) * 1000)
        elif eraseMode == 2:
            return datetime.datetime.utcnow().replace(microsecond=0)

    @classmethod
    def dateDiff(cls, dtStart, dtEnd=None):
        """
        Get difference in millis between two datetime
        :param dtStart: Start datetime
        :type dtStart: datetime
        :param dtEnd: End datetime (will use current utc if not provided)
        :type dtEnd: datetime
        :return float
        :rtype float
        """

        # Get delta
        if dtEnd:
            # noinspection PyUnresolvedReferences
            delta = dtEnd - dtStart
        else:
            # noinspection PyTypeChecker
            delta = cls.dateCurrent() - dtStart

        # From : http://stackoverflow.com/questions/4898687/python-parsing-timestamps-and-calculating-time-differences-in-milliseconds
        return ((delta.days * 86400 + delta.seconds) * 1000) + (delta.microseconds * 0.001)

    #===============================
    # COMPO NAME (FOR RSYSLOG)
    #===============================

    @classmethod
    def setCompoName(cls, compoName):
        """
        Set the component name. Useful for rsyslog.
        :param compoName: The component name or None. If None, method do nothing.
        :type compoName: str,None
        """

        if compoName:
            cls._compoName = compoName
            lifecyclelogger.info("compoName now set to=%s", cls._compoName)

    @classmethod
    def getCompoName(cls):
        """
        Get current component name.
        :return str
        :rtype str
        """

        return cls._compoName

    @classmethod
    def getMachineName(cls):
        """
        Get machine name
        :return: Machine name
        :rtype: str
        """

        return platform.uname()[1]

    #===============================
    # MISC
    #===============================

    @classmethod
    def sleep(cls, sleepMs):
        """
        Sleep for specified ms.
        Also used as gevent context switch in code, since it rely on gevent.sleep.
        :param sleepMs: Millis to sleep.
        :type sleepMs: int
        :return Nothing.
        """
        gevent.sleep(sleepMs * 0.001)

    #===============================
    # EXCEPTION HELPER
    #===============================

    @classmethod
    def exToStr(cls, e, maxLevel=10, maxPathLevel=5):
        """
        Format an exception.
        :param e: Any exception instance.
        :type e: Exception
        :param maxLevel: Maximum call stack level (default 10)
        :type maxLevel: int
        :param maxPathLevel: Maximum path level (default 5)
        :type maxPathLevel: int
        :return The exception readable string
        :rtype str
        """

        # Go
        listFrame = None
        try:
            outBuffer = ""

            # Class type
            outBuffer += "e.cls:[{0}]".format(e.__class__.__name__)

            # To string
            try:
                exBuf = str(e)
            except UnicodeEncodeError:
                exBuf = str(repr(unicode(e)))
            except Exception as e:
                logger.error("Exception, e=%s", e)
                raise
            outBuffer += ", e.str:[{0}]".format(exBuf)

            # Traceback
            si = sys.exc_info()

            # Raw frame
            # tuple : (file, lineno, method, code)
            rawFrame = traceback.extract_tb(si[2])
            rawFrame.reverse()

            # Go to last tb_next
            last_tb_next = None
            cur_tb = si[2]
            while cur_tb:
                last_tb_next = cur_tb
                cur_tb = cur_tb.tb_next

            # Skip frame up to current raw frame count
            listFrame = list()
            curCount = -1
            skipCount = len(rawFrame)
            curFrame = last_tb_next.tb_frame
            while curFrame:
                curCount += 1
                if curCount < skipCount:
                    curFrame = curFrame.f_back
                else:
                    # Need : tuple : (file, lineno, method, code)
                    rawFrame.append((curFrame.f_code.co_filename, curFrame.f_lineno, curFrame.f_code.co_name, ""))
                    curFrame = curFrame.f_back

            # Build it
            curIdx = 0
            outBuffer += ", e.cs=["
            for tu in rawFrame:
                line = str(tu[1])
                curfile = tu[0]
                method = tu[2]

                # Handle max path level
                arToken = curfile.rsplit(os.sep, maxPathLevel)
                if len(arToken) > maxPathLevel:
                    # Remove head
                    arToken.pop(0)
                    # Join
                    curfile = "..." + os.sep.join(arToken)

                # Format
                outBuffer += "in:{0}#{1}@{2} ".format(method, curfile, line)

                # Loop
                curIdx += 1
                if curIdx >= maxLevel:
                    outBuffer += "..."
                    break

            # Close
            outBuffer += "]"

            # Ok
            return outBuffer
        finally:
            if listFrame:
                del listFrame

    #===============================
    # VOODOO INIT
    #===============================

    @classmethod
    def voodooInit(cls, aggressive=True):
        """
        Global initialization, to call asap.
        Apply gevent stuff & default logging configuration.
        :param aggressive: bool
        :type aggressive: bool
        :return Nothing.
        """

        # Check
        if cls._voodooInitialized:
            return

        # Lock
        with cls._voodooLock:
            # Re-check
            if cls._voodooInitialized:
                return

            # Fire the voodoo magic :)
            lifecyclelogger.info("Voodoo : gevent : entering, aggressive=%s", aggressive)
            monkey.patch_all(aggressive=aggressive)
            lifecyclelogger.info("Voodoo : gevent : entering")

            # Initialize log level to INFO
            lifecyclelogger.info("Voodoo : logging : entering")
            cls.loggingInit()
            lifecyclelogger.info("Voodoo : logging : done")

            # Done
            cls._voodooInitialized = True

    #===============================
    # LOGGING
    #===============================

    @classmethod
    def loggingInit(cls, logLevel="INFO", forceReset=False, logCallback=None, logToFile=None):
        """
        Initialize logging sub system with default settings (console, pre-formatted output)
        :param logLevel: The log level to set. Any value in "DEBUG", "INFO", "WARN", "ERROR", "CRITICAL"
        :type logLevel: str
        :param forceReset: If true, logging system is reset.
        :type forceReset: bool
        :param logToFile: If specified, log to file
        :type logToFile: str
        :param logCallback: Callback for unittest
        :return Nothing.
        """

        if cls._loggingInitialized == True and forceReset == False:
            return

        with cls._loggingLock:
            if cls._loggingInitialized == True and forceReset == False:
                return

            # Default
            logging.basicConfig(level=logLevel)

            # Formatter
            f = logging.Formatter(
                "%(asctime)s | %(levelname)s | %(module)s@%(funcName)s@%(lineno)d | %(message)s | %(thread)d:%(threadName)s | %(process)d:%(processName)s")

            # Console handler
            c = logging.StreamHandler(sys.stdout)
            c.setLevel(logging.getLevelName(logLevel))
            c.setFormatter(f)

            # File handler to /tmp
            cf=None
            if logToFile:
                cf = logging.FileHandler(logToFile)
                cf.setLevel(logging.getLevelName(logLevel))
                cf.setFormatter(f)

            # Syslog handler
            try:
                from sol.SysLogger import SysLogger

                syslog = SysLogger(logCallback=logCallback)
                syslog.setLevel(logging.getLevelName(logLevel))
                syslog.setFormatter(f)
            except Exception as e:
                raise Exception("Unable to import SysLogger, e=%s", SolBase.exToStr(e))

            # Initialize
            root = logging.getLogger()
            root.setLevel(logging.getLevelName(logLevel))
            root.handlers = []
            root.addHandler(c)
            if logToFile:
                root.addHandler(cf)
            root.addHandler(syslog)


            # Done
            cls._loggingInitialized = True
            lifecyclelogger.info("Logging : initialized from memory, logLevel=%s, forceReset=%s", logLevel, forceReset)

    @classmethod
    def loggingInitFomFile(cls, configFileName, forceReset=False):
        """
        Initialize logging system from a configuration file, with optional reset.
        :param configFileName: Configuration file name
        :type configFileName: str
        :param forceReset: If true, logging system is reset.
        :type forceReset: bool
        :return Nothing.
        """

        if cls._loggingInitialized == True and forceReset == False:
            return

        with cls._loggingLock:
            if cls._loggingInitialized == True and forceReset == False:
                return

            try:
                logger.debug("Logging : configFileName=%s", configFileName)
                fileConfig(configFileName, None, False)
                lifecyclelogger.info("Logging : initialized from file, configFileName=%s", configFileName)
            except Exception as e:
                logger.error("Exception, e=%s", cls.exToStr(e))
                raise

    #===============================
    # FORK STUFF
    #===============================

    @classmethod
    def getMasterProcess(cls):
        """
        Return True if we are the master process, False otherwise.
        :return bool
        :rtype bool
        """
        return cls._masterProcess

    @classmethod
    def setMasterProcess(cls, isMaster):
        """
        Set is we are a fork master or not
        :param isMaster: True if we are master process, False if we are a child process.
        :type isMaster: bool
        :return Nothing
        """

        logger.info("Switching _masterProcess to %s", isMaster)
        cls._masterProcess = isMaster

    #===============================
    # BINARY STUFF
    #===============================

    @classmethod
    def binaryToUnicode(cls, binBuf, encoding="utf-8"):
        """
        Binary buffer to unicode, using the specified encoding
        :param binBuf: Binary buffer
        :type binBuf: str
        :param encoding: Encoding to use
        :type encoding: str
        :return unicode
        :rtype unicode
        """

        return unicode(binBuf, encoding)

    @classmethod
    def unicodeToBinary(cls, unicodeBuf, encoding="utf-8"):
        """
        Unicode to binary buffer, using the specified encoding
        :param unicodeBuf: String to convert.
        :type unicodeBuf: unicode
        :param encoding: Encoding to use.
        :type encoding: str
        :return str
        :rtype str
        """

        return unicodeBuf.encode(encoding)

    #===============================
    # CONVERSIONS
    #===============================

    @classmethod
    def toInt(cls, v):
        """
        Convert to int
        :param v: int,str
        :type v: int,str
        :return: int
        :rtype int
        """

        if isinstance(v, int):
            return v
        else:
            return int(v)

    @classmethod
    def toLong(cls, v):
        """
        Convert to long
        :param v: long,str
        :type v: long,str
        :return: long
        :rtype long
        """

        if isinstance(v, long):
            return v
        else:
            return long(v)

    @classmethod
    def toBool(cls, v):
        """
        Convert to bool
        :param v: bool,str
        :type v: bool,str
        :return: bool
        :rtype bool
        """

        if isinstance(v, bool):
            return v
        else:
            return ast.literal_eval(v)

    @classmethod
    def getClassName(cls, myInstance):
        """
        Return the class name of myInstance, or "Instance.None".
        :param cls: Our class.
        :param myInstance: Instance to use.
        :return: Return the class name of myInstance, or "Instance.None" in case of error/None value.
        """
        if myInstance is None:
            return "Instance.None"
        else:
            return myInstance.__class__.__name__

    @classmethod
    def getPathSeparator(cls):
        """
        Return the path separator.
        http://docs.python.org/library/os.html#os.sep
        :param cls: Our class
        :return: The path separator (string)
        """
        return os.sep

    @classmethod
    def isString(cls, myString):
        """
        Return true if the provided myString is a str or an unicode.
        :param cls: Our class.
        :param myString: A String.
        :return: Return true if the provided myString is a str or an unicode. False otherwise.
        """
        if myString is None:
            return False
        else:
            return isinstance(myString, (str, unicode))

    @classmethod
    def isStringNotEmpty(cls, myString):
        """
        Return true if the provided myString is a str or an unicode, not empty.
        :param cls: Our class.
        :param myString: A String.
        :return: Return true if the provided myString is a str or an unicode, not empty.. False otherwise.
        """
        if not SolBase.isString(myString):
            return False
        else:
            return len(myString) > 0

    @classmethod
    def isBoolean(cls, myBool):
        """
        Return true if the provided myBool is a boolean.
        :param cls: Our class.
        :param myBool: A boolean..
        :return: Return true if the provided myBool is a boolean. False otherwise.
        """
        if myBool is None:
            return False
        else:
            return isinstance(myBool, bool)

    @classmethod
    def isInteger(cls, myInt):
        """
        Return true if the provided myInt is a integer.
        :param cls: Our class.
        :param myInt: An integer..
        :return: Return true if the provided myInt is a integer. False otherwise.
        """
        if myInt is None:
            return False
        # Caution, boolean is an integer...
        elif SolBase.isBoolean(myInt):
            return False
        else:
            return isinstance(myInt, (int, long))

    @classmethod
    def getCurrentPidsAsString(cls):
        """
        Return the current pids as string.
        :param cls: Our class.
        :return: A String
        """
        return "pid={0}, ppid={1}".format(os.getpid(), os.getppid())


