"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""

# Logger
import logging
import gevent
from gevent.queue import Empty
from sol.Meter.MeterManager import MeterManager
from sol.SolBase import SolBase
from sol.TcpBase.SignaledBuffer import SignaledBuffer
from sol.TcpBase.TcpSocketManager import TcpSocketManager
from sol.TcpServer.TcpServerStat import TcpServerStat

SolBase.loggingInit()
logger = logging.getLogger("TcpServerClientContext")


class TcpServerClientContext(TcpSocketManager):
    """
    Tcp server client context.
    """

    def __init__(self, tcpServer, clientId, clientSocket, clientAddr, cbStopAsynch=None, cbOnReceive=None):
        """
        Constructor.
        :param tcpServer: The tcp server instance.
        :param clientId: The client id.
        :param clientSocket: The client socket.
        :param clientAddr: The client remote address.
        :param cbStopAsynch: Callback to call upon stop. If None, self.stopAsynch is used.
        :param cbOnReceive: Callback to call upon socket receive. If none, self._onReceive is used.
        :return Nothing.
        """

        # Check
        if clientSocket is None:
            logger.error("TcpServerClientContext : __init__ : clientSocket is None")
            raise Exception("TcpServerClientContext : __init__ : clientSocket is None")
        elif clientAddr is None:
            logger.error("TcpServerClientContext : __init__ : clientAddr is None")
            raise Exception("TcpServerClientContext : __init__ : clientAddr is None")
        elif clientId is None:
            logger.error("TcpServerClientContext : __init__ : clientId is None")
            raise Exception("TcpServerClientContext : __init__ : clientId is None")
        elif tcpServer is None:
            logger.error("TcpServerClientContext : __init__ : tcpServer is None")
            raise Exception("TcpServerClientContext : __init__ : tcpServer is None")

        # Base - we provide two callback :
        # - one for disconnecting ourselves
        # - one to notify socket receive buffer

        # Stop callback
        curCbStopAsynch = cbStopAsynch
        if cbStopAsynch is None:
            # Use default
            curCbStopAsynch = self.stopAsynch

        # Receive callback
        curCbOnReceive = cbOnReceive
        if curCbOnReceive is None:
            # Use default
            curCbOnReceive = self._onReceive

        TcpSocketManager.__init__(self, curCbStopAsynch, curCbOnReceive)

        # Store
        self._clientId = clientId
        self._clientAddr = clientAddr

        # Base store
        self.currentSocket = clientSocket

        # Server store
        self._tcpServer = tcpServer

        # Greenlets
        self._readGreenlet = None
        self._writeGreenlet = None
        self._controlGreenlet = None

        # Faster str
        self._socketLocalIp = None
        self._socketLocalPort = None
        self._socketRemoteIp = None
        self._socketRemotePort = None

    #================================
    # TO STRING OVERWRITE
    #================================

    def __getSocketLocalIp(self):
        """
        Get socket local ip
        :return: String
        :rtype: str
        """
        if not self._socketLocalIp is None:
            return self._socketLocalIp
        try:
            self._socketLocalIp = self.currentSocket.getsockname()[0]
            return self._socketLocalIp
        except:
            return "exception"

    def __getSocketLocalPort(self):
        """
        Get socket local ip
        :return: String
        :rtype: str
        """

        if not self._socketLocalPort is None:
            return self._socketLocalPort
        try:
            self._socketLocalPort = self.currentSocket.getsockname()[1]
            return self._socketLocalPort
        except:
            return "exception"

    def __getSocketRemoteIp(self):
        """
        Get socket remote ip
        :return: String
        :rtype: str
        """
        if not self._socketRemoteIp is None:
            return self._socketRemoteIp
        try:
            self._socketRemoteIp = self.currentSocket.getpeername()[0]
            return self._socketRemoteIp
        except:
            return "exception"

    def __getSocketRemotePort(self):
        """
        Get socket remote port
        :return: String
        :rtype: str
        """
        if not self._socketRemotePort is None:
            return self._socketRemotePort

        try:
            self._socketRemotePort = self.currentSocket.getpeername()[1]
            return self._socketRemotePort
        except:
            return "exception"

    def __str__(self):
        """
        To string override
        :return: A string
        :rtype string
        """

        return "c.id={0}*c.cli={1}:{2}/{3}:{4}*{5}".format(
            self._clientId,
            self.__getSocketLocalIp(),
            self.__getSocketLocalPort(),
            self.__getSocketRemoteIp(),
            self.__getSocketRemotePort(),
            TcpSocketManager.__str__(self)
        )

    #===============================
    # START
    #===============================

    def start(self):
        """
        Start processing our socket read/write.
        :return True if success, false otherwise.
        """
        try:
            # Check
            if self.isConnected:
                logger.warn("TcpServerClientContext : start : already connected, doing nothing, self=%s", self)
                return False

            # Done
            self.isConnected = True
            logger.debug("TcpServerClientContext : start : now connected, starting r/w loops, self=%s", self)

            # Start the read/write loops
            self._readGreenlet = gevent.spawn(self._readLoop)
            self._writeGreenlet = gevent.spawn(self._writeLoop)

            # Start the control greenlet
            self._scheduleControlGreenlet()

            # Done
            logger.debug("TcpServerClientContext : start : done, self=%s", self)
            return True
        except Exception as e:
            logger.error("TcpServerClientContext : start : Exception, ex=%s, self=%s", SolBase.exToStr(e), self)
            return False

    #===============================
    # STOP
    #
    # - stopAsynch
    # => used ONLY if stop comes from OUR instance (read/write loop initiated stop, due to socket issues, control stuff)
    # => This delegates the stop to the tcpServer, which will process it asynchronously, ending with a stopSynch call.
    #
    # - stopSynch
    # => used by external components to request a stop.
    #===============================

    def stopAsynch(self):
        """
        Request a stop to the tcpServer.. It will call _stopInternal(), via tcpServer._removeClientAsynch.
        Do NOT OVERRIDE this method.
        :return True if success, false otherwise.
        """
        try:
            if not self.isConnected:
                return True
            else:
                # noinspection PyProtectedMember
                self._tcpServer._removeClientAsynch(self.getClientId())
                return True
        except Exception as e:
            logger.error("TcpServerClientContext : stopFromServer : Exception, ex=%s, self=%s",
                         SolBase.exToStr(e), self)
            return False

    def stopSynch(self):
        """
        Stop processing our socket read/write.
        CAUTION : This method NOT be called if the server is stopping.
        Do NOT call this from ourselves, instead use stopAsynch, you may/will experience unexpected behaviors.
        This method is RESERVED for EXTERNAL class stop requests.
        You can OVERRIDE this method at HIGHER level to be informed of a socket closure.
        :return True if success, false otherwise.
        """

        return True

    def stopSynchInternal(self):
        """
        Stop processing our socket read/write.
        Reserved for PURE in-memory stop operations (greenlet stop, counter put mainly)
        NEVER, NEVER perform any kind of non-memory operations here.
        For instance, are FORDIDEN in higher level implementation of stopSynchInternal :
        - Any socket send/recv
        - Any external queries (redis/mongo, whatever)
        JUST HANDLE IN MEMORY STUFF.
        :return True if success, false otherwise.
        """

        try:
            logger.debug("TcpServerClientContext : disconnect : entering, self=%s", self)

            # Check
            if not self.isConnected:
                logger.debug("TcpServerClientContext : disconnect : not connected, doing nothing, self=%s", self)
                return False

            # Signal (move on top, try to avoid some TcpManager warn logs while stopping)
            self.isConnected = False

            # Timeout unschedule
            self._unscheduleSslHandshakeTimeOut()

            # Control unschedule
            self._unscheduleControlGreenlet()

            # Disconnect
            # Close the socket in this case (should not cover mantis 1173)
            TcpSocketManager.safeCloseSocket(self.currentSocket)
            self.currentSocket = None

            # Greenlet reset after isConnected=False (will help to exit itself)
            if not self._readGreenlet is None:
                self._readGreenlet.kill()
                self._readGreenlet = None

            if not self._writeGreenlet is None:
                self._writeGreenlet.kill()
                self._writeGreenlet = None

            # Flush out the send queue now, and decrement pending bytes to send
            totalLen = 0
            while True:
                try:
                    item = self.sendQueue.get(False)
                    if isinstance(item, str):
                        totalLen += len(item)
                    elif isinstance(item, SignaledBuffer):
                        totalLen += len(item.binaryBuffer)
                except Empty:
                    break

            # Decrement
            logger.debug("TcpServerClientContext : disconnect : decrementing, totalLen=%s", totalLen)
            MeterManager.get(TcpServerStat).serverBytesSendPending.increment(-totalLen)

            # Over
            logger.debug("TcpServerClientContext : disconnect : done, self=%s", self)
            return True

        except Exception as e:
            logger.error("TcpServerClientContext : disconnect : Exception, ex=%s, self=%s", SolBase.exToStr(e),
                         self)
            return False
        finally:
            # Session duration stats
            sec = SolBase.dateDiff(self._dtCreated) / 1000
            MeterManager.get(TcpServerStat).sessionDurationSecond.put(sec)

    #===============================
    # GETTER / SETTER
    #===============================

    def getClientId(self):
        """
        Getter
        :return The client id.
        """
        return self._clientId

    def getClientSocket(self):
        """
        Getter
        :return The client socket.
        """
        return self.currentSocket

    def getClientAddr(self):
        """
        Getter
        return: The client remote address.
        """
        return self._clientAddr

    #===============================
    # RECEIVE
    #===============================

    def _onReceive(self, binaryBuffer):
        """
        Called on socket receive.
        :param binaryBuffer: The received buffer.
        :return Nothing.
        """

        # Got something
        logger.debug("TcpServerClientContext : _onReceive called, binaryBuffer=%s, self=%s", repr(binaryBuffer), self)
        pass

    #===============================
    # CONTROL
    #===============================

    def _scheduleControlGreenlet(self):
        """
        Schedule the control greenlet for next run.
        Note: We do not use lock to minimize per socket memory usage...
        :return: Nothing.
        """

        # Check
        if not self.isConnected:
            return

        # Get
        ms = self._tcpServer.getEffectiveControlIntervalMs()
        if ms <= 0:
            return

        # Schedule it !
        self._controlGreenlet = gevent.spawn_later(ms * 0.001, self._controlUs)


    def _unscheduleControlGreenlet(self):
        """
        unschedule the control greenlet.
        Note: We do not use lock to minimize per socket memory usage...
        :return: Nothing.
        """

        if not self._controlGreenlet is None:
            self._controlGreenlet.kill(block=False)
            self._controlGreenlet = None

    def _controlUs(self):
        """
        Control us. May force ourself to exit upon inactivity.
        Note: We do not use lock to minimize per socket memory usage...
        :return: Nothing.
        """

        logger.info("Entering")

        # Check
        if not self.isConnected:
            return

        # Get settings
        c = self._tcpServer.getTcpServerConfig()

        # Check absolute
        absoluteMs = c.socketAbsoluteTimeOutMs
        if absoluteMs > 0:
            runningMs = SolBase.dateDiff(self._dtCreated)
            if runningMs > absoluteMs:
                logger.debug("Absolute reached, runningMs=%s, absoluteMs=%s, self=%s", runningMs, absoluteMs, self)
                # Kill ourself if we are running
                if self.isConnected:
                    self.stopAsynch()
                return
            else:
                logger.debug("Absolute not reached, runningMs=%s, absoluteMs=%s, self=%s", runningMs, absoluteMs, self)
                pass

        # Check relative
        relativeMs = c.socketRelativeTimeOutMs
        if relativeMs > 0:
            # Get last receive ms
            lastRecvMs = SolBase.dateDiff(self._dtLastRecv)

            # Get last send ms
            lastSendMs = SolBase.dateDiff(self._dtLastSend)

            # We keep the minimum of both
            lastMs = min(lastRecvMs, lastSendMs)

            # Check
            if lastMs > relativeMs:
                # Kill ourself
                logger.debug("Relative reached, lastRecvMs=%s, lastSendMs=%s, lastMs=%s, relativeMs=%s, self=%s",
                             lastRecvMs, lastSendMs, lastMs, relativeMs, self)
                if self.isConnected:
                    self.stopAsynch()
                return
            else:
                logger.debug("Relative not reached, lastRecvMs=%s, lastSendMs=%s, lastMs=%s, relativeMs=%s, self=%s",
                             lastRecvMs, lastSendMs, lastMs, relativeMs, self)
                pass

        # Schedule next check
        self._scheduleControlGreenlet()


