#!/usr/bin/env python

"""
A "builder" for Veracity's distributed build tracking feature.
"""

import os
import sys
import time
import argparse
import tempfile
import subprocess
from collections import defaultdict
import logging

from veracity import BUILDER, VERSION
from veracity.objects import _BaseRepositoryObject
from veracity import common
from veracity import settings


class Builder(_BaseRepositoryObject):
    """Represents a builder for a particular repository."""

    def __init__(self, repo, config):
        """Represent a tag.
        @param repo: repository name
        @param config: common Config object
        """
        super(Builder, self).__init__(repo=repo)
        self.config = config
        self.count = 0
        self.stats = defaultdict(int)

    def __repr__(self):
        return self._repr(self.repo, self.config)

    def __str__(self):
        return "'{r}' builder".format(r=self.repo)

    def __eq__(self, other):
        if not isinstance(other, Builder):
            return False
        else:
            return all((super(Builder, self).__eq__(other),
                        self.config.environments == other.config.environments))

    def __ne__(self, other):
        return not self == other

    def run(self):
        """Perform one build iteration.
        """
        for environment in self.config.environments:
            for status in self.config.starts:

                # Look for a build request
                request = self.repo.get_build(environment, status)
                if not request:
                    time.sleep(settings.BUILD_DELAY)
                    continue

                # Check out build request's changeset
                logging.info("found build request: {0}".format(request))
                self.count += 1
                with self.repo.checkout(tempfile.mkdtemp(), rev=request.cid) as work:
                    # Update the configuration from the changeset
                    path = common.find_config(work.path)
                    if path:  # pragma: no cover - the test repo does not have a config
                        logging.info("updating the build configuration from the changeset...")
                        self.config.update(path)
                    # Build state machine
                    while True:
                        # Enter the next build state
                        request.update(self.config.enter(request.status))
                        if request.status in self.config.ends:  # pragma: no cover - test build does not end normally
                            break
                        # Run the command for the current state
                        path, command = self.config.get_build(request.status)
                        work.chdir(path)
                        success = self.call(command)
                        # Process the command results
                        if success:
                            request.update(self.config.exit(request.status))
                        else:
                            request.update(self.config.fail(request.status))
                        if request.status in self.config.ends:
                            break
                self.stats[request.status] += 1
                logging.info("completed builds: {0}".format(self.count))

    def test(self, single=None):
        """Perform one build iteration locally as a test.
        """
        for status in ([single] if single else self.config.starts):
            while True:
                # Enter the next build state
                status = self.config.enter(status)
                if status in self.config.ends:  # pragma: no cover - the test repo does not have a config
                    break
                # Run the command for the current state
                path, command = self.config.get_build(status)
                os.chdir(path)
                success = self.call(command)
                # Process the command results
                if success:
                    status = self.config.exit(status)
                else:
                    status = self.config.fail(status)
                if single or status in self.config.ends:
                    break

    @staticmethod
    def call(command):
        """Run a build command in the current directory.
        @param command: shell string
        @return: indicates command returned no error
        """
        if command:
            logging.info("$ {0}".format(command))
            return subprocess.call(command, shell=True) == 0
        else:
            logging.info("$ (no command)")
            return True


def run(config, daemon=False, test=False):
    """Run a builder for each repository.
    @param config: common Config object
    @param daemon: run as a daemon
    @param test: run locally as a test
    @return: indicates builder(s) ran successfully
    """
    builders = [Builder(repo, config) for repo in config.repos]
    if not builders:
        return False

    while True:
        for builder in builders:
            if isinstance(test, basestring):
                logging.info("testing [{s}] of {b}...".format(b=builder, s=test))
                builder.test(single=test)
            elif test:
                logging.info("testing {b}...".format(b=builder))
                builder.test()
            else:
                logging.log(logging.DEBUG if daemon else logging.INFO, "running {b}...".format(b=builder))
                builder.run()
        if not daemon or test:
            break

    return True


# TODO: share the common logic with tracking, builder, and poller
def main():  # pragma: no cover - tested manually
    """Process command-line arguments and run the program.
    """
    # Main parser
    parser = argparse.ArgumentParser(prog=BUILDER, description=__doc__, formatter_class=common.formatter_class)
    parser.add_argument('-V', '--version', action='version', version=VERSION)
    parser.add_argument('-v', '--verbose', action='count', help="enable verbose logging")
    parser.add_argument('-d', '--daemon', action='store_true', help="keep the process running")
    parser.add_argument('-t', '--test', metavar='ALIAS', nargs='?', const=True, help="dry run locally as a test")
    parser.add_argument('repo', nargs='*', help="repository names")
    parser.add_argument('--config', metavar='PATH', help="path to a configuration file")
    parser.add_argument('--env', metavar='ALIAS', help="environment alias for this machine")

    # Parse arguments
    args = parser.parse_args()

    # Configure logging
    common.configure_logging(args.verbose)

    # Parse the configuration file
    config = common.parse_config(args.config, parser.error, repos=args.repo, envs=[args.env] if args.env else None)
    if not config.repos:
        parser.error("specify at least one repository")
    if len(config.environments) != 1 and not args.test:
        parser.error("specify exactly one environment alias for this machine")

    # Ensure we are running as the correct user
    common.check_user(args.daemon, parser.error)

    # Run the program
    success = False
    try:
        success = run(config, args.daemon, args.test)
    except KeyboardInterrupt:  # pylint: disable=W0703
        logging.debug("cancelled manually")
    if not success:
        sys.exit(1)


if __name__ == '__main__':  # pragma: no cover - tested manually
    main()
