"""
# -*- coding: utf-8 -*-
#====================================================================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#====================================================================================================================
"""

from _socket import SOL_SOCKET, SOL_TCP, SO_KEEPALIVE, TCP_KEEPIDLE, TCP_KEEPINTVL, TCP_KEEPCNT
from _ssl import PROTOCOL_TLSv1
import logging
from threading import Lock

from gevent import GreenletExit
import gevent
from gevent.queue import Queue
import gevent.socket as gsocket
from gevent.ssl import SSLSocket
import socks

from pythonsol.AtomicInt import AtomicInt
from pythonsol.SolBase import SolBase
from pythonsol.TcpBase.ProtocolParserTextDelimited import ProtocolParserTextDelimited
from pythonsol.TcpBase.TcpSocketManager import TcpSocketManager
from pythonsol.TcpClient.TcpClientConfig import TcpClientConfig


SolBase.loggingInit()
logger = logging.getLogger("TcpSimpleClient")


class TcpSimpleClient(TcpSocketManager):
    """
    Tcp simple client
    """

    def __init__(self, tcpClientConfig, onReceiveProfileSelf=False):
        """
        Constructor.
        Option via TcpClientConfig to enable SSL on socket.
        In this case, SSL handshake is to be done manually AFTER connect. Higher level implementation MUST handle this.
        :param tcpClientConfig: The tcp client config.
        :param onReceiveProfileSelf: If True, self will be provided as onReceive param
        :type onReceiveProfileSelf: bool
        :return Nothing.
        """

        # Base - we provide two callback :
        # - one for disconnecting ourselves
        # - one to notify socket receive buffer
        TcpSocketManager.__init__(self, self.disconnect, self._onReceive, onReceiveProfileSelf=onReceiveProfileSelf)

        # Check
        if tcpClientConfig is None:
            logger.error("TcpSimpleClient : __init__ : tcpServerConfig is None")
            raise Exception("TcpSimpleClient : __init__ : tcpServerConfig is None")
        elif not isinstance(tcpClientConfig, TcpClientConfig):
            logger.error(
                "TcpSimpleClient : __init__ : tcpServerConfig is not a TcpServerConfig, class=%s",
                SolBase.getClassName(tcpClientConfig))
            raise Exception("TcpSimpleClient : __init__ : tcpServerConfig is not a TcpServerConfig")

        # Default
        self._tcpClientConfig = tcpClientConfig

        # Greenlets
        self._readGreenlet = None
        self._writeGreenlet = None

        # Receive queue
        self._receiveQueue = Queue()

        # Receive current buffer
        self._receiveCurrentBuf = None

        # Socket info
        self._localAdr = None
        self._localPort = None
        self._remoteAdr = None
        self._remotePort = None
        self._connectCount = AtomicInt()
        self._disconnectCount = AtomicInt()

        # Pool stuff
        self.processId = None
        self.poolId = None
        self.dtAlloc = SolBase.dateCurrent()
        self.dtLastAcquire = self.dtAlloc
        self.dtLastRelease = self.dtAlloc
        self.countAcquire = 0
        self.countRelease = 0

        # Pool disconnect notify
        self.onDisconnectNotifyPoolCallbackLock = Lock()
        self.onDisconnectNotifyPoolCallbackCall = True
        self.onDisconnectNotifyPoolCallback = None


    #================================
    # POOL METHOD
    #================================

    def notifyAcquire(self):
        """
        Notify
        """
        self.dtLastAcquire = SolBase.dateCurrent()
        self.countAcquire += 1

    def notifyRelease(self):
        """
        Notify
        """
        self.dtLastRelease = SolBase.dateCurrent()
        self.countRelease += 1


    #================================
    # TO STRING OVERWRITE
    #================================

    def __str__(self):
        """
        To string override
        :return: A string
        :rtype string
        """

        return "c.addr={0}:{1}/{2}:{3}*cc={4}*dc={5}*c.q.recv.size={6}*sock={7}*proxy={8}/{9}:{10}*{11}".format(
            self._localAdr, self._localPort, self._remoteAdr, self._remotePort,
            self._connectCount.get(), self._disconnectCount.get(),
            self._receiveQueue.qsize(),
            self.currentSocket,
            self._tcpClientConfig.proxyEnable,
            self._tcpClientConfig.proxyAddr,
            self._tcpClientConfig.proxyPort,
            TcpSocketManager.__str__(self)
        )

    #=================================
    # CONNECT
    #=================================

    def connect(self):
        """
        Connect to server. Return true upon success.
        If SSL is enable, do NOT forget at higher level to initiate a SSL handshake manually.
        :return True if connected, false otherwise.
        """

        try:
            logger.debug("TcpSimpleClient : connect : starting, target=%s:%s", self._tcpClientConfig.targetAddr,
                         self._tcpClientConfig.targetPort)

            # Reset the error flag right now
            self.unsetInternalFatalErrorForReconnect()

            # Check
            if self.isConnected == True:
                logger.warn("TcpSimpleClient : connect : already connected, doing nothing, self=%s", self)
                return False


            # Init
            self._connectCount.increment()
            self._dtCreated = SolBase.dateCurrent()
            self._dtLastRecv = self._dtCreated
            self._dtLastSend = self._dtCreated

            #------------------------
            # ALLOC
            #------------------------
            if self._tcpClientConfig.sslEnable == True:
                if self._tcpClientConfig.proxyEnable == True:
                    #------------------------
                    # PROXY ON, SSL ON
                    #------------------------
                    if self._tcpClientConfig.debugLog == True:
                        logger.info("TcpSimpleClient : Alloc : PROXY ON/SSL ON, self=%s", self)
                    else:
                        logger.debug("TcpSimpleClient : Alloc : PROXY ON/SSL ON, self=%s", self)
                        pass

                    # Alloc
                    self.currentSocket = socks.socksocket()

                    # Proxy
                    self.currentSocket.setproxy(socks.PROXY_TYPE_SOCKS5, self._tcpClientConfig.proxyAddr,
                                                self._tcpClientConfig.proxyPort)

                    # SSL Wrap : Inside POST CONNECT
                else:
                    #------------------------
                    # PROXY OFF, SSL ON
                    #------------------------
                    if self._tcpClientConfig.debugLog == True:
                        logger.info("TcpSimpleClient : Alloc : PROXY OFF/SSL ON, self=%s", self)
                    else:
                        logger.debug("TcpSimpleClient : Alloc : PROXY OFF/SSL ON, self=%s", self)
                        pass

                    # Alloc
                    self.current = gevent.socket.socket()

                    # Wrap
                    self.currentSocket = SSLSocket(self.currentSocket, do_handshake_on_connect=False,
                                                   ssl_version=PROTOCOL_TLSv1)
            else:
                if self._tcpClientConfig.proxyEnable == True:
                    #------------------------
                    # PROXY ON, SSL OFF
                    #------------------------
                    if self._tcpClientConfig.debugLog == True:
                        logger.info("TcpSimpleClient : Alloc : PROXY ON/SSL OFF, self=%s", self)
                    else:
                        logger.debug("TcpSimpleClient : Alloc : PROXY ON/SSL OFF, self=%s", self)
                        pass

                    # Alloc
                    self.currentSocket = socks.socksocket()

                    # Proxy
                    self.currentSocket.setproxy(socks.PROXY_TYPE_SOCKS5, self._tcpClientConfig.proxyAddr,
                                                self._tcpClientConfig.proxyPort)
                else:
                    #------------------------
                    # PROXY OFF, SSL OFF
                    #------------------------
                    if self._tcpClientConfig.debugLog == True:
                        logger.info("TcpSimpleClient : Alloc : PROXY OFF/SSL OFF, self=%s", self)
                    else:
                        logger.debug("TcpSimpleClient : Alloc : PROXY OFF/SSL OFF, self=%s", self)
                        pass

                    # Alloc
                    self.currentSocket = gsocket.socket()

            #------------------------
            # POST ALLOC
            #------------------------
            if not self._tcpClientConfig.timeoutMs is None:
                logger.debug("TcpSimpleClient : connect : setting timeout=%s, self=%s", self._tcpClientConfig.timeoutMs,
                             self)
                self.currentSocket.settimeout(self._tcpClientConfig.timeoutMs * 0.001)
                logger.debug("TcpSimpleClient : connect : timeout=%s, self=%s", self.currentSocket.gettimeout(), self)

            #------------------------
            # CONNECT
            #------------------------

            # 2-tuple (host, port)
            address = (self._tcpClientConfig.targetAddr, self._tcpClientConfig.targetPort)

            # Connect
            if self._tcpClientConfig.debugLog == True:
                logger.info("TcpSimpleClient : connect : starting, ssl=%s, timeout=%s, self=%s",
                            self._tcpClientConfig.sslEnable,
                            self._tcpClientConfig.timeoutMs, self)

            # Go
            logger.debug("TcpSimpleClient : connect : calling connect, self=%s", self)
            self.currentSocket.connect(address)
            logger.debug("TcpSimpleClient : connect : now connected, self=%s", self)

            # Store
            self._localAdr = self.currentSocket.getsockname()[0]
            self._localPort = self.currentSocket.getsockname()[1]
            self._remoteAdr = self.currentSocket.getpeername()[0]
            self._remotePort = self.currentSocket.getpeername()[1]

            # Log
            if self._tcpClientConfig.debugLog == True:
                logger.info("TcpSimpleClient : connected, %s:%s to %s:%s, self=%s", self._localAdr, self._localPort,
                            self._remoteAdr, self._remotePort, self)

            # Done
            self.isConnected = True
            self.onDisconnectNotifyPoolCallbackCall = True

            #------------------------
            # POST CONNECT
            #------------------------
            if self._tcpClientConfig.sslEnable == True:
                # Switch to SSL now
                if self._tcpClientConfig.debugLog == True:
                    logger.info("TcpSimpleClient : PostConnect : PROXY ON/SSL ON, switching to SSL now, self=%s", self)
                else:
                    logger.debug("TcpSimpleClient : PostConnect : PROXY ON/SSL ON, switching to SSL now, self=%s", self)

                # Wrap (already connected
                self.currentSocket = SSLSocket(self.currentSocket, do_handshake_on_connect=False,
                                               ssl_version=PROTOCOL_TLSv1)

                # Start
                dtStart = SolBase.dateCurrent()

                # Do it
                logger.debug("TcpSimpleClient : __doSslHandshake now, self=%s", self)
                self.__doSslHandshake()

                # Time
                self._setSslHandshakeMs(SolBase.dateDiff(dtStart))

            #------------------------
            # POST CONNECT
            #------------------------

            if self._tcpClientConfig.tcpKeepAliveEnabled == True:
                # Switch to TCP KA now
                if self._tcpClientConfig.debugLog == True:
                    logger.info(
                        "TcpSimpleClient : PostConnect : TCP KA ON, switching to KA now, (on=%s/delay=%s/failed=%s/interval=%s), self=%s",
                        self._tcpClientConfig.tcpKeepAliveEnabled,
                        self._tcpClientConfig.tcpKeepAliveProbesSendDelayMs,
                        self._tcpClientConfig.tcpKeepAliveProbesFailedCount,
                        self._tcpClientConfig.tcpKeepAliveProbesSendIntervalMs,
                        self)
                else:
                    logger.debug(
                        "TcpSimpleClient : PostConnect : TCP KA ON, switching to KA now, (on=%s/delay=%s/failed=%s/interval=%s), self=%s",
                        self._tcpClientConfig.tcpKeepAliveEnabled,
                        self._tcpClientConfig.tcpKeepAliveProbesSendDelayMs,
                        self._tcpClientConfig.tcpKeepAliveProbesFailedCount,
                        self._tcpClientConfig.tcpKeepAliveProbesSendIntervalMs,
                        self)

                # Go
                self.currentSocket.setsockopt(SOL_SOCKET, SO_KEEPALIVE, 1)
                self.currentSocket.setsockopt(SOL_TCP, TCP_KEEPIDLE,
                                              self._tcpClientConfig.tcpKeepAliveProbesSendDelayMs / 1000)
                self.currentSocket.setsockopt(SOL_TCP, TCP_KEEPINTVL,
                                              self._tcpClientConfig.tcpKeepAliveProbesSendIntervalMs / 1000)
                self.currentSocket.setsockopt(SOL_TCP, TCP_KEEPCNT, self._tcpClientConfig.tcpKeepAliveProbesFailedCount)

                # Check
                v = self.currentSocket.getsockopt(SOL_SOCKET, SO_KEEPALIVE)
                if v != 1:
                    logger.warn("SO_KEEPALIVE mismatch, having=%s, required=1", v)

                v = self.currentSocket.getsockopt(SOL_TCP, TCP_KEEPIDLE)
                if v != self._tcpClientConfig.tcpKeepAliveProbesSendDelayMs / 1000:
                    logger.warn("TCP_KEEPIDLE mismatch, having=%s, required=%s", v,
                                self._tcpClientConfig.tcpKeepAliveProbesSendDelayMs / 1000)

                v = self.currentSocket.getsockopt(SOL_TCP, TCP_KEEPINTVL)
                if v != self._tcpClientConfig.tcpKeepAliveProbesSendIntervalMs / 1000:
                    logger.warn("TCP_KEEPINTVL mismatch, having=%s, required=%s", v,
                                self._tcpClientConfig.tcpKeepAliveProbesSendIntervalMs / 1000)

                v = self.currentSocket.getsockopt(SOL_TCP, TCP_KEEPCNT)
                if v != self._tcpClientConfig.tcpKeepAliveProbesFailedCount:
                    logger.warn("TCP_KEEPCNT mismatch, having=%s, required=%s", v,
                                self._tcpClientConfig.tcpKeepAliveProbesFailedCount)

            # Non-blocking mode
            self.currentSocket.setblocking(0)
            logger.debug("TcpSimpleClient : connect : non blocking mode set, self=%s", self)

            # Start the read/write loops
            self._readGreenlet = gevent.spawn(self._readLoop)
            self._writeGreenlet = gevent.spawn(self._writeLoop)
            logger.debug("TcpSimpleClient : connect : r/w loops started, self=%s", self)

            # Done
            logger.debug("TcpSimpleClient : connect : done, self=%s", self)
            return True

        except Exception as e:
            # Logs
            logger.error("TcpSimpleClient : connect : Exception, ex=%s, self=%s", SolBase.exToStr(e), self)

            # Call disconnect
            self._callbackDisconnect()

            # Exit
            return False

    #=================================
    # DISCONNECT
    #=================================

    def disconnect(self):
        """
        Disconnect from server. Return true upon success.
        :return True if success, false otherwise.
        """
        try:
            logger.debug("TcpSimpleClient : disconnect : entering, self=%s", self)

            # Check
            if self.isConnected == False:
                logger.debug("TcpSimpleClient : disconnect : not connected, doing nothing, self=%s", self)
                return False

            # Disconnect
            if not self.currentSocket is None:
                logger.debug("TcpSimpleClient : disconnect : socket.shutdown, self=%s", self)
                #noinspection PyUnusedLocal
                try:
                    self.currentSocket.shutdown(2)
                except Exception as e:
                    logger.debug("Exception on shutdown, ex=%s, self=%s", SolBase.exToStr(e), self)
                    pass

                logger.debug("TcpSimpleClient : disconnect : socket.close, self=%s", self)
                self.currentSocket.close()
                self.currentSocket = None

            # Reset socket related context
            self._dtCreated = SolBase.dateCurrent()
            self._dtLastRecv = self._dtCreated
            self._dtLastSend = self._dtCreated
            self._disconnectCount.increment()
            self._localAdr = None
            self._localPort = None
            self._remoteAdr = None
            self._remotePort = None

            # Signal
            logger.debug("TcpSimpleClient : disconnect : isConnected=False, self=%s", self)
            self.isConnected = False

            # Handle notifications at upper level
            goCall = False

            # Second level callback - Process in lock to avoid multiple notification for same simple client instance
            with self.onDisconnectNotifyPoolCallbackLock:
                if self.onDisconnectNotifyPoolCallbackCall == True and self.onDisconnectNotifyPoolCallback:
                    logger.debug(
                        "TcpSimpleClient : disconnect : calling second level disconnect callback, callback=%s, self=%s",
                        self.onDisconnectNotifyPoolCallback, self)
                    self.onDisconnectNotifyPoolCallbackCall = False
                    goCall = True

            # Out of lock : call (ASYNC, because we may be called by read/write greenlet, which are going to be killed just after)
            if goCall == True:
                try:
                    # Call it synchronously (mantis 1907)
                    self.onDisconnectNotifyPoolCallback(self)

                    # DO NOT MOVE THIS SLEEP, REDIS WILL DEADLOCK ON START
                    SolBase.sleep(0)
                except Exception as e:
                    logger.warn("onDisconnectNotifyPoolCallback : call exception=%s", SolBase.exToStr(e))

            # Greenlet reset after isConnected=False (will help to exit itself)
            if not self._readGreenlet is None:
                logger.debug("TcpSimpleClient : disconnect : read kill, self=%s", self)
                self._readGreenlet.kill(GreenletExit, False)
                logger.debug("TcpSimpleClient : disconnect : read kill 1, self=%s", self)
                self._readGreenlet = None
                logger.debug("TcpSimpleClient : disconnect : read kill 2, self=%s", self)

            if not self._writeGreenlet is None:
                logger.debug("TcpSimpleClient : disconnect : write kill, self=%s", self)
                self._writeGreenlet.kill(GreenletExit, False)
                logger.debug("TcpSimpleClient : disconnect : write kill 1, self=%s", self)
                self._writeGreenlet = None
                logger.debug("TcpSimpleClient : disconnect : write kill 2, self=%s", self)

            logger.debug("TcpSimpleClient : disconnect : done, self=%s", self)
            return True

        except Exception as e:
            logger.error("TcpSimpleClient : disconnect : Exception, ex=%s, self=%s", SolBase.exToStr(e), self)
            return False

    #=================================
    # SSL HANDSHAKE
    #=================================

    def __doSslHandshake(self):
        """
        Do a ssl handshake. Method is using a greenlet, but is blocking for the caller.
        :return: Nothing.
        """

        # Go
        SolBase.sleep(0)
        sslGreenlet = gevent.spawn(self.__doSslHandshakeInternal)

        # Wait for complete
        SolBase.sleep(0)
        sslGreenlet.join()

    def __doSslHandshakeInternal(self):
        """
        Do a ssl handshake. Call _callbackDisconnect() if error.
        :return: Nothing
        """
        try:
            logger.debug("__doSslHandshakeInternal : start now, self=%s", self)
            SolBase.sleep(0)
            self.currentSocket.do_handshake()
        except Exception as e:
            logger.error("__doSslHandshakeInternal : exception=%s, self=%s", SolBase.exToStr(e), self)
            self._callbackDisconnect()
        finally:
            SolBase.sleep(0)

    #=================================
    # RECEIVE
    #=================================

    def _onReceive(self, binaryBuffer):
        """
        Callback called upon server receive.
        - binaryBuffer : a BINARY buffer received on the socket.

        !! CAUTION !!
        - IN ALL CASES THIS METHOD RECEIVE A BINARY BUFFER

        !! CAUTION !!
        - PYTHON 2.X / binary - str    => str is a byte array (or an ascii string) : nothing to do
        - PYTHON 2.X / text - unicode  => you receive a str, you may convert it to unicode (SolBase.binaryToUnicode)

        !! CAUTION !!
        - PYTHON 3.X / binary - bytes  => bytes is binary. You may convert it to a str using encoding.

        DOC
        - http://stackoverflow.com/questions/8219706/difference-between-binary-string-byte-string-unicode-string-and-an-ordinary-st

                |  2.x                     |  3.x
        --------+--------------------------+-----------------------
        Bytes   |  'abc' <type 'str'>      |  b'abc' <type 'bytes'>
        Unicode | u'abc' <type 'unicode'>  |   'abc' <type 'str'>
        :param binaryBuffer: Binary buffer received.
        :return: Nothing.
        """

        # Received something...
        logger.debug("TcpSimpleClient : _onReceive : binaryBuffer=%s, self=%s", repr(binaryBuffer), self)

        # Parse
        self._receiveCurrentBuf = ProtocolParserTextDelimited.parseProtocol(self._receiveCurrentBuf,
                                                                            binaryBuffer, self._receiveQueue, "\n")


    def getRecvQueueLen(self):
        """
        Get receive queue len
        :return The receive queue length.
        """
        return self._receiveQueue.qsize()

    #=================================
    # RECEIVE QUEUE
    #=================================

    def getFromReceiveQueue(self, block=False, timeOutSec=None):
        """
        Get a buffer from the receive queue.
        - If block is False, will return an item OR raise an Empty exception if no item.
        - If block is True AND timeOut=None, will wait forever for an item.
        - If block is True and timeOutSec>0, will wait for timeOutSec then raise Empty exception if no item.
        :param block: If true, will block.
        :param timeOutSec: The timeout in second.
        """

        return self._receiveQueue.get(block, timeOutSec)



