"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
#
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""
from threading import Lock


class AtomicInt(object):
    """
    Atomic integer without lock.
    """

    UINT64_MAX = 18446744073709551615

    def __init__(self, initialValue=0, maximumValue=UINT64_MAX):
        """
        Init
        :param initialValue: Initial value
        :type initialValue: int
        :param maximumValue: Maximum value (default : UINT64)
        :type maximumValue: int
        :return Nothing
        """

        self._currentValue = initialValue
        self._maximumValue = maximumValue
        self._processMaxValue()

    def increment(self, incrementValue=1):
        """
        Increment and return current value
        :param incrementValue: Increment value
        :type incrementValue: int
        :return Current value
        :rtype int
        """

        self._currentValue += incrementValue
        self._processMaxValue()
        return self._currentValue

    def set(self, currentValue):
        """
        Set current value
        :param currentValue: New current value
        :type currentValue: int
        :return Current value
        :rtype int
        """

        self._currentValue = currentValue
        self._processMaxValue()
        return self._currentValue

    def get(self):
        """
        Get current value
        :return Current value
        :rtype int
        """
        return self._currentValue

    def _processMaxValue(self):
        """
        Process max value if required
        :return Nothing
        """

        if self._maximumValue and self._currentValue > self._maximumValue:
            # Compute & reset with reminder if maximum value reached
            div = long(self._currentValue / self._maximumValue)
            reminder = self._currentValue % self._maximumValue
            if div > 0:
                if reminder == 0:
                    self._currentValue = self._maximumValue
                else:
                    self._currentValue = reminder
            else:
                self._currentValue = reminder


class AtomicIntSafe(AtomicInt):
    """
    Safe atomic integer with lock.
    """

    def __init__(self, initialValue=0, maximumValue=AtomicInt.UINT64_MAX):
        """
        Init
        :param initialValue: Initial value
        :type initialValue: int
        :param maximumValue: Maximum value (default : UINT64)
        :type maximumValue: int
        :return Nothing
        """

        self._lock = Lock()
        with self._lock:
            AtomicInt.__init__(self, initialValue, maximumValue)

    def increment(self, incrementValue=1):
        """
        Increment and return current value
        :param incrementValue: Increment value
        :type incrementValue: int
        :return Current value
        :rtype int
        """

        with self._lock:
            return AtomicInt.increment(self, incrementValue)

    def set(self, currentValue):
        """
        Set current value
        :param currentValue: New current value
        :type currentValue: int
        :return Current value
        :rtype int
        """

        with self._lock:
            return AtomicInt.set(self, currentValue)

    def get(self):
        """
        Get current value
        :return Current value
        :rtype int
        """
        with self._lock:
            return AtomicInt.get(self)
