"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""

# Import
import logging
import os
import codecs
from pythonsol.SolBase import SolBase

SolBase.loggingInit()
logger = logging.getLogger("FileUtility")


class FileUtility(object):
    """
    Class description
    Open doc : http://docs.python.org/py3k/library/functions.html#open
    Encoding doc : http://docs.python.org/py3k/library/locale.html#locale.getpreferredencoding
    Python 3 is internally unicode for "str" type.
    """

    def __init__(self):
        """
        Constructor.
        :return: Nothing.
        """

    @staticmethod
    def isPathExist(pathName):
        """
        Check if a path (file or dir) name exist.
        :param pathName: Path name.
        :return: Return true (exist), false (do not exist, or invalid file name)
        """

        #  Check
        if pathName is None:
            logger.error("isPathExist : fileName is None")
            return False
        elif not isinstance(pathName, str):
            logger.error("isPathExist : pathName not a str, className=%s", SolBase.getClassName(pathName))
            return False

        # Go
        return os.path.exists(pathName)

    @staticmethod
    def isFileExist(fileName):
        """
        Check if file name exist.
        :param fileName: File name.
        :return: Return true (exist), false (do not exist, or invalid file name)
        """

        #  Check
        if fileName is None:
            logger.error("isFileExist : fileName is None")
            return False
        elif not isinstance(fileName, str):
            logger.error("isFileExist : fileName not a str, className=%s", SolBase.getClassName(fileName))
            return False

        # Go
        return os.path.isfile(fileName)

    @staticmethod
    def isDirExist(dirName):
        """
        Check if dir name exist.
        :param dirName: Directory name.
        :return: Return true (exist), false (do not exist, or invalid file name)
        """

        #  Check
        if dirName is None:
            logger.error("isDirExist : fileName is None")
            return False
        elif not isinstance(dirName, str):
            logger.error("isDirExist : fileName not a str, className=%s", SolBase.getClassName(dirName))
            return False

        # Go
        return os.path.isdir(dirName)

    @staticmethod
    def getFileSize(fileName):
        """
        Return a file size in bytes.
        :param fileName: File name.
        :return: An integer, gt-eq 0 if file exist, lt 0 if error.
        """
        if not FileUtility.isFileExist(fileName):
            return -1
        else:
            return os.path.getsize(fileName)

    @classmethod
    def getCurrentDir(cls):
        """
        Return the current directory.
        :return: A String
        """

        return os.getcwd()

    @classmethod
    def splitPath(cls, path):
        """
        Split path, returning a list.
        :param path: A path.
        :return: A list of path component.
        """
        return path.split(SolBase.getPathSeparator())

    @staticmethod
    def fileToByteBuffer(fileName):
        """
        Load a file toward a binary buffer.
        :param fileName: File name.
        :return: Return the binary buffer or None in case of error.
        """

        # Check
        if not FileUtility.isFileExist(fileName):
            logger.error("fileToByteBuffer : fileName not exist, fileName=%s", fileName)
            return None

        # Go
        rd = None
        try:
            # Open (binary : open return a io.BufferedReader)
            rd = open(fileName, "rb")

            # Preallocate buffer (faster)
            binbuf = bytearray(FileUtility.getFileSize(fileName))

            # Read everything
            #noinspection PyArgumentList
            rd.readinto(binbuf)

            # Return
            return binbuf
        except IOError as e:
            # Exception...
            logger.error("fileToByteBuffer : IOError, ex=%s", SolBase.exToStr(e))
            return None
        except Exception as e:
            logger.error("fileToByteBuffer : Exception, ex=%s", SolBase.exToStr(e))
            return None
        finally:
            # Close if not None...
            if not rd is None:
                rd.close()

    @staticmethod
    def fileToTextBuffer(fileName, encoding):
        """
        Load a file toward a text buffer (UTF-8), using the specify encoding while reading.
        CAUTION : This will read the whole file IN MEMORY.
        :param fileName: File name.
        :param encoding: Encoding to user.
        :return: A text buffer or None in case of error.
        """

        # Check
        if not FileUtility.isFileExist(fileName):
            logger.error("fileToTextBuffer : fileName not exist, fileName=%s", fileName)
            return None

        # Go
        rd = None
        try:
            # Open (text : open return a io.BufferedReader)
            rd = codecs.open(fileName, "rt", encoding, "strict", -1)

            # Read everything
            return rd.read()
        except IOError as e:
            # Exception...
            logger.error("fileToByteBuffer : IOError, ex=%s", SolBase.exToStr(e))
            return None
        except Exception as e:
            logger.error("fileToByteBuffer : Exception, ex=%s", SolBase.exToStr(e))
            return None
        finally:
            # Close if not None...
            if not rd is None:
                rd.close()

    @staticmethod
    def appendBinaryToFile(fileName, binaryBuffer):
        """
        Write to the specified filename, the provided binary buffer.
        Create the file if required.
        :param fileName:  File name.
        :param binaryBuffer: Binary buffer to write.
        :return: The number of bytes written or lt 0 if error.
        """

        # Go
        rd = None
        try:
            # Open (text : open return a io.BufferedReader)
            rd = open(fileName, "ab+")

            # Read everything
            return rd.write(binaryBuffer)
        except IOError as e:
            # Exception...
            logger.error("appendBinaryToFile : IOError, ex=%s", SolBase.exToStr(e))
            return -1
        except Exception as e:
            logger.error("appendBinaryToFile : Exception, ex=%s", SolBase.exToStr(e))
            return -1
        finally:
            # Close if not None...
            if not rd is None:
                rd.close()

    @staticmethod
    def appendTextToFile(fileName, textBuffer, encoding, overwrite=False):
        """
        Write to the specified filename, the provided binary buffer
        Create the file if required.
        :param fileName:  File name.
        :param textBuffer: Text buffer to write.
        :param encoding: The encoding to user.
        :param overwrite: If true, file is overwritten.
        :return: The number of bytes written or lt 0 if error.
        """

        # Go
        rd = None
        try:
            # Open (text : open return a io.BufferedReader)
            if not overwrite:
                rd = codecs.open(fileName, "at+", encoding, "strict", -1)
            else:
                rd = codecs.open(fileName, "wt", encoding, "strict", -1)

            # Read everything
            # CAUTION : 2.7 return None :(
            return rd.write(textBuffer)
        except IOError as e:
            # Exception...
            logger.error("appendTextToFile : IOError, ex=%s", SolBase.exToStr(e))
            return -1
        except Exception as e:
            logger.error("appendTextToFile : Exception, ex=%s", SolBase.exToStr(e))
            return -1
        finally:
            # Close if not None...
            if not rd is None:
                rd.close()
