"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""

# Logger
import logging
from pythonsol.FileUtility import FileUtility
from pythonsol.SolBase import SolBase
from pythonsol.TcpServer.BaseFactory.TcpServerClientContextAbstractFactory import TcpServerClientContextAbstractFactory
from pythonsol.TcpServer.ClientContext.TcpServerClientContextFactory import TcpServerClientContextFactory

SolBase.loggingInit()
logger = logging.getLogger("TcpServerConfig")


class TcpServerConfig(object):
    """
    TcpServer configuration.
    If ssl is NOT enable, it is better that calls to sslCertificate/sslKeyFile should not happen.
    """

    def __init__(self):
        """
        Constructor.
        """

        logger.debug("TcpServerConfig : Constructor called")

        # Default
        self._autoStart = False
        self._listenAddr = None
        self._listenPort = None
        self._setClientFactory(TcpServerClientContextFactory())

        # Ssl
        self._sslEnable = False
        self._sslKeyFile = None
        self._sslCertificateFile = None
        self._sslHandshakeTimeOutMs = 10000

        # Stop behavior
        self._onStopCallClientStopSynch = False

        # Forking
        self._childProcessCount = 0

        # Debug
        self._debugWaitInSslMs = None

        # Socket control : Absolute default = 7 days (x 24 hours, x60 min, x60 sec, x1000 for ms)
        self._socketAbsoluteTimeOutMs = 7 * 24 * 60 * 60 * 1000

        # Socket control : Relative default = 1 days (x 24 hours, x60 min, x60 sec, x1000 for ms)
        self._socketRelativeTimeOutMs = 1 * 24 * 60 * 60 * 1000

        # Socket control : minimal interval between 2 check = 60 sec
        self._socketMinCheckIntervalMs = 60000

        # Timeout while stopping client
        self._stopClientTimeoutMs = 10000

        # Timeout while stopping server
        self._stopServerTimeoutMs = 30000

    def _setClientFactory(self, clientFactory):
        """
        Setter. Raise exception if a problem occurs.
        :param clientFactory : must be an instance of the factory, not the class itself.
        :return Nothing.
        """
        logger.debug("TcpServerConfig : _setClientFactory called, clientFactory=%s, class=%s", clientFactory,
                     SolBase.getClassName(clientFactory))

        if clientFactory is None:
            logger.error("TcpServerConfig : _setClientFactory : clientFactory none")
            raise Exception("TcpServerConfig : _setClientFactory : clientFactory none")
        elif not isinstance(clientFactory, TcpServerClientContextAbstractFactory):
            logger.error(
                "TcpServerConfig : _setClientFactory : clientFactory is not a TcpServerClientContextAbstractFactory")
            raise Exception(
                "TcpServerConfig : _setClientFactory : clientFactory is not a TcpServerClientContextAbstractFactory")
        else:
            self._clientFactory = clientFactory

    def _setListenAddr(self, listenAddr):
        """
        Setter. Raise exception if a problem occurs.
        :param listenAddr: The listen address.
        :return Nothing.
        """
        logger.debug("TcpServerConfig : _setListenAddr called, listenAddr=%s, class=%s", listenAddr,
                     SolBase.getClassName(listenAddr))

        if not SolBase.isStringNotEmpty(listenAddr):
            logger.error("TcpServerConfig : _setListenAddress : not a string or empty, class=%s",
                         SolBase.getClassName(listenAddr))
            raise Exception("TcpServerConfig : _setListenAddress : not a string or empty")
        else:
            self._listenAddr = listenAddr

    def _setListenPort(self, inListenPort):
        """
        Setter. Raise exception if a problem occurs.
        :param inListenPort: The listen port.
        :return Nothing.
        """

        listenPort = inListenPort
        logger.debug("TcpServerConfig : _setListenPort called, listenPort=%s, class=%s", listenPort,
                     SolBase.getClassName(listenPort))

        if SolBase.isString(listenPort):
            listenPort = SolBase.toInt(listenPort)

        if not SolBase.isInteger(listenPort):
            logger.error("TcpServerConfig : _setListenPort : not a int, class=%s", SolBase.getClassName(listenPort))
            raise Exception("TcpServerConfig : _setListenPort : not a int")
        elif listenPort == 0:
            logger.warn("TcpServerConfig : _setListenPort : newPort==0")
            raise Exception("TcpServerConfig : _setListenPort : newPort==0")
        else:
            self._listenPort = listenPort

    def _setSslHandshakeTimeOutMs(self, inTimeOutMs):
        """
        Setter. Raise exception if a problem occurs.
        :param inTimeOutMs: Time out in ms.
        :return Nothing.
        """

        timeOutMs = inTimeOutMs
        logger.debug("TcpServerConfig : _setSslHandshakeTimeOutMs called, timeOutMs=%s, class=%s", timeOutMs,
                     SolBase.getClassName(timeOutMs))

        if SolBase.isString(timeOutMs):
            timeOutMs = SolBase.toInt(timeOutMs)

        if not SolBase.isInteger(timeOutMs):
            logger.error("TcpServerConfig : _setSslHandshakeTimeOutMs : not a int, class=%s",
                         SolBase.getClassName(timeOutMs))
            raise Exception("TcpServerConfig : _setSslHandshakeTimeOutMs : not a int")
        else:
            self._sslHandshakeTimeOutMs = timeOutMs

    def _setStopClientTimeoutMs(self, inTimeOutMs):
        """
        Setter. Raise exception if a problem occurs.
        :param inTimeOutMs: Time out in ms.
        :return Nothing.
        """

        timeOutMs = inTimeOutMs
        logger.debug("TcpServerConfig : _setStopClientTimeoutMs called, timeOutMs=%s, class=%s", timeOutMs,
                     SolBase.getClassName(timeOutMs))

        if SolBase.isString(timeOutMs):
            timeOutMs = SolBase.toInt(timeOutMs)

        if not SolBase.isInteger(timeOutMs):
            logger.error("TcpServerConfig : _setStopClientTimeoutMs : not a int, class=%s",
                         SolBase.getClassName(timeOutMs))
            raise Exception("TcpServerConfig : _setStopClientTimeoutMs : not a int")
        else:
            self._stopClientTimeoutMs = timeOutMs

    def _setStopServerTimeoutMs(self, inTimeOutMs):
        """
        Setter. Raise exception if a problem occurs.
        :param inTimeOutMs: Time out in ms.
        :return Nothing.
        """

        timeOutMs = inTimeOutMs
        logger.debug("TcpServerConfig : _setStopServerTimeoutMs called, timeOutMs=%s, class=%s", timeOutMs,
                     SolBase.getClassName(timeOutMs))

        if SolBase.isString(timeOutMs):
            timeOutMs = SolBase.toInt(timeOutMs)

        if not SolBase.isInteger(timeOutMs):
            logger.error("TcpServerConfig : _setStopServerTimeoutMs : not a int, class=%s",
                         SolBase.getClassName(timeOutMs))
            raise Exception("TcpServerConfig : _setStopServerTimeoutMs : not a int")
        else:
            self._stopServerTimeoutMs = timeOutMs

    def _setChildProcessCount(self, inChildProcessCount):
        """
        Setter. Raise exception if a problem occurs.
        :param inChildProcessCount: The number of child process count. If zero, no fork is performed (default).
        :return Nothing.
        """

        childProcessCount = inChildProcessCount
        logger.debug("TcpServerConfig : _setChildProcessCount called, childProcessCount=%s, class=%s",
                     childProcessCount,
                     SolBase.getClassName(childProcessCount))

        if SolBase.isString(childProcessCount):
            childProcessCount = SolBase.toInt(childProcessCount)

        if not SolBase.isInteger(childProcessCount):
            logger.error("TcpServerConfig : _setChildProcessCount : not a int, class=%s",
                         SolBase.getClassName(childProcessCount))
            raise Exception("TcpServerConfig : _setChildProcessCount : not a int")
        elif childProcessCount < 0:
            logger.warn("TcpServerConfig : _setChildProcessCount : childProcessCount<0")
            raise Exception("TcpServerConfig : _setChildProcessCount : childProcessCount<0")
        else:
            self._childProcessCount = childProcessCount

    def _setAutoStart(self, inMyBool):
        """
        Getter
        :param inMyBool: A boolean.
        """

        myBool = inMyBool
        logger.debug("TcpServerConfig : _setAutoStart called, myBool=%s, class=%s", myBool,
                     SolBase.getClassName(myBool))

        if SolBase.isString(myBool):
            myBool = SolBase.toBool(myBool)

        if not SolBase.isBoolean(myBool):
            logger.error("TcpServerConfig : _setAutoStart : not a boolean, class=%s", SolBase.getClassName(myBool))
            raise Exception("TcpServerConfig : _setAutoStart : not a boolean")
        else:
            self._autoStart = myBool

    def _setOnStopCallClientStopSynch(self, inMyBool):
        """
        Getter
        :param inMyBool: A boolean.
        """

        myBool = inMyBool
        logger.debug("TcpServerConfig : _setOnStopCallClientStopSynch called, myBool=%s, class=%s", myBool,
                     SolBase.getClassName(myBool))

        if SolBase.isString(myBool):
            myBool = SolBase.toBool(myBool)

        if not SolBase.isBoolean(myBool):
            logger.error("TcpServerConfig : _setOnStopCallClientStopSynch : not a boolean, class=%s",
                         SolBase.getClassName(myBool))
            raise Exception("TcpServerConfig : _setOnStopCallClientStopSynch : not a boolean")
        else:
            self._onStopCallClientStopSynch = myBool


    def _setSslEnable(self, inIsEnable):
        """
        Enable or disable ssl.
        :param inIsEnable: Boolean.
        :return: Nothing.
        """

        isEnable = inIsEnable
        logger.debug("TcpServerConfig : _setSslEnable called, isEnable=%s, class=%s", isEnable,
                     SolBase.getClassName(isEnable))

        if SolBase.isString(isEnable):
            isEnable = SolBase.toBool(isEnable)

        if not SolBase.isBoolean(isEnable):
            logger.error("TcpServerConfig : _setSslEnable : not a boolean, class=%s", SolBase.getClassName(isEnable))
            raise Exception("TcpServerConfig : _setSslEnable : not a boolean")
        else:
            self._sslEnable = isEnable

    def _setSslKeyFile(self, keyFile):
        """
        Set the ssl key file. Raise exception upon error.
        :param keyFile: A valid key file. Must exist and be accessible.
        :return: True upon success, false otherwise.
        """

        logger.debug("TcpServerConfig : _setSslKeyFile called, keyFile=%s, class=%s", keyFile,
                     SolBase.getClassName(keyFile))

        # Check
        if keyFile is None or (SolBase.isString(keyFile) == True and len(keyFile) == 0):
            self._sslKeyFile = ""
            return

        if not SolBase.isStringNotEmpty(keyFile):
            logger.warn("TcpServerConfig : _setSslKeyFile : not a string, class=%s", SolBase.getClassName(keyFile))
            raise Exception("TcpServerConfig : _setSslKeyFile : not a string")

        if not FileUtility.isFileExist(keyFile):
            logger.warn("_setSslKeyFile : keyFile do not exist or is not accessible, keyFile=%s", keyFile)

        # Set
        self._sslKeyFile = keyFile

    def _setSslCertificateFile(self, certFile):
        """
        Set the ssl certificate file. Raise exception upon error.
        :param certFile: A valid certificate file. Must exist and be accessible.
        :return: Nothing.
        """

        logger.debug("TcpServerConfig : _setSslCertificateFile called, keyFile=%s, class=%s", certFile,
                     SolBase.getClassName(certFile))

        # Check
        if certFile is None or (SolBase.isString(certFile) == True and len(certFile) == 0):
            self._sslCertificateFile = ""
            return

        if not SolBase.isStringNotEmpty(certFile):
            logger.error("TcpServerConfig : _setSslCertificateFile : not a string, class=%s",
                         SolBase.getClassName(certFile))
            raise Exception("TcpServerConfig : _setSslCertificateFile : not a string")

        # Check
        if not FileUtility.isFileExist(certFile):
            logger.warn("_setSslCertificateFile : certFile do not exist or is not accessible, certFile=%s", certFile)

        # Set
        self._sslCertificateFile = certFile

    def _getClientFactory(self):
        """
        Getter
        :return The client factory.
        """
        return self._clientFactory

    def _getListenPort(self):
        """
        Getter
        :return The listen port.
        """
        return self._listenPort

    def _getSslHandshakeTimeOutMs(self):
        """
        Getter
        :return The value.
        """
        return self._sslHandshakeTimeOutMs

    def _getStopClientTimeoutMs(self):
        """
        Getter
        :return The value.
        """
        return self._stopClientTimeoutMs

    def _getStopServerTimeoutMs(self):
        """
        Getter
        :return The value.
        """
        return self._stopServerTimeoutMs


    def _getChildProcessCount(self):
        """
        Getter
        :return The number of child process count.
        """
        return self._childProcessCount

    def _getAutoStart(self):
        """
        Getter
        :return The auto start boolean
        """
        return self._autoStart

    def _getOnStopCallClientStopSynch(self):
        """
        Getter
        :return The auto start boolean
        """
        return self._onStopCallClientStopSynch

    def _getListenAddr(self):
        """
        Getter
        :return The listen address.
        """
        return self._listenAddr

    def _getSslEnable(self):
        """
        Getter.
        :return: A boolean.
        """
        return self._sslEnable

    def _getSslKeyFile(self):
        """
        Getter.
        :return: A string or None
        """

        return self._sslKeyFile

    def _getSslCertificateFile(self):
        """
        Getter.
        :return: A string or None
        """

        return self._sslCertificateFile

    def _getSocketAbsoluteTimeOutMs(self):
        """
        Getter
        :return Timeout in ms.
        """
        return self._socketAbsoluteTimeOutMs

    def _getSocketRelativeTimeOutMs(self):
        """
        Getter
        :return Timeout in ms.
        """
        return self._socketRelativeTimeOutMs

    def _getSocketMinCheckIntervalMs(self):
        """
        Getter
        :return Timeout in ms.
        """
        return self._socketMinCheckIntervalMs

    def _setSocketMinCheckIntervalMs(self, inCheckMs):
        """
        Setter. Raise exception if a problem occurs.
        :param inCheckMs: Timeout in ms.
        :return Nothing.
        """

        checkMs = inCheckMs
        logger.debug("TcpServerConfig : _setSocketMinCheckIntervalMs called, checkMs=%s, class=%s",
                     checkMs,
                     SolBase.getClassName(checkMs))

        if SolBase.isString(checkMs):
            checkMs = SolBase.toInt(checkMs)

        if not SolBase.isInteger(checkMs):
            logger.error("TcpServerConfig : _setSocketMinCheckIntervalMs : not a int, class=%s",
                         SolBase.getClassName(checkMs))
            raise Exception("TcpServerConfig : _setSocketMinCheckIntervalMs : not a int")
        else:
            self._socketMinCheckIntervalMs = checkMs

    def _setSocketAbsoluteTimeOutMs(self, inTimeOutMs):
        """
        Setter. Raise exception if a problem occurs.
        :param inTimeOutMs: Timeout in ms.
        :return Nothing.
        """

        timeOutMs = inTimeOutMs
        logger.debug("TcpServerConfig : _setSocketAbsoluteTimeOutMs called, timeOutMs=%s, class=%s",
                     timeOutMs,
                     SolBase.getClassName(timeOutMs))

        if SolBase.isString(timeOutMs):
            timeOutMs = SolBase.toInt(timeOutMs)

        if not SolBase.isInteger(timeOutMs):
            logger.error("TcpServerConfig : _setSocketAbsoluteTimeOutMs : not a int, class=%s",
                         SolBase.getClassName(timeOutMs))
            raise Exception("TcpServerConfig : _setSocketAbsoluteTimeOutMs : not a int")
        else:
            self._socketAbsoluteTimeOutMs = timeOutMs

    def _setSocketRelativeTimeOutMs(self, inTimeOutMs):
        """
        Setter. Raise exception if a problem occurs.
        :param inTimeOutMs: Timeout in ms.
        :return Nothing.
        """

        timeOutMs = inTimeOutMs
        logger.debug("TcpServerConfig : _setSocketRelativeTimeOutMs called, timeOutMs=%s, class=%s",
                     timeOutMs,
                     SolBase.getClassName(timeOutMs))

        if SolBase.isString(timeOutMs):
            timeOutMs = SolBase.toInt(timeOutMs)

        if not SolBase.isInteger(timeOutMs):
            logger.error("TcpServerConfig : _setSocketRelativeTimeOutMs : not a int, class=%s",
                         SolBase.getClassName(timeOutMs))
            raise Exception("TcpServerConfig : _setSocketRelativeTimeOutMs : not a int")
        else:
            self._socketRelativeTimeOutMs = timeOutMs

    #========================
    # Properties - configs
    #========================

    # Auto-start on ?
    autoStart = property(_getAutoStart, _setAutoStart)
    # Listen address
    listenAddr = property(_getListenAddr, _setListenAddr)
    # Listen port
    listenPort = property(_getListenPort, _setListenPort)
    # Client factory
    clientFactory = property(_getClientFactory, _setClientFactory)
    # Ssl on ?
    sslEnable = property(_getSslEnable, _setSslEnable)
    # Ssl on ?
    onStopCallClientStopSynch = property(_getOnStopCallClientStopSynch, _setOnStopCallClientStopSynch)
    # Ssl key file
    sslKeyFile = property(_getSslKeyFile, _setSslKeyFile)
    # Ssl certificate file
    sslCertificateFile = property(_getSslCertificateFile, _setSslCertificateFile)

    #========================
    # Properties - tuning
    #========================

    # Child process count
    childProcessCount = property(_getChildProcessCount, _setChildProcessCount)
    # Timeout in ms allowed for SSL handshake
    sslHandshakeTimeOutMs = property(_getSslHandshakeTimeOutMs, _setSslHandshakeTimeOutMs)
    # Per socket : absolute socket time out in ms. If lt or eq to 0 : no limit.
    socketAbsoluteTimeOutMs = property(_getSocketAbsoluteTimeOutMs, _setSocketAbsoluteTimeOutMs)
    # Per socket : relative socket time out in ms. If lt or eq to 0 : no limit.
    socketRelativeTimeOutMs = property(_getSocketRelativeTimeOutMs, _setSocketRelativeTimeOutMs)
    # Socket : Minimum check interval in ms.
    socketMinCheckIntervalMs = property(_getSocketMinCheckIntervalMs, _setSocketMinCheckIntervalMs)
    # Stop timeout ms
    stopClientTimeoutMs = property(_getStopClientTimeoutMs, _setStopClientTimeoutMs)
    # Stop timeout ms
    stopServerTimeoutMs = property(_getStopServerTimeoutMs, _setStopServerTimeoutMs)

    #========================
    # Properties - DEBUG ONLY - NEVER USE THEM IN PRODUCTION
    #========================

    def _setDebugWaitInSslMs(self, value):
        """
        Setter for debug only.
        :param value: Value.
        :return: Nothing.
        """
        self._debugWaitInSslMs = value

    def _getDebugWaitInSslMs(self):
        """
        Getter for debug only.
        :return: Value.
        """
        return self._debugWaitInSslMs

    debugWaitInSslMs = property(_getDebugWaitInSslMs, _setDebugWaitInSslMs)


