"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
#
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""
import glob
import logging
import os
from threading import Lock

from gevent import GreenletExit
import gevent

from pythonsol.SolBase import SolBase


logger = logging.getLogger("MeterManager")
lifecyclelogger = logging.getLogger("LifeCycle")


class MeterManager(object):
    """
    Static meter manager.
    """

    # Global lock
    _staticLock = Lock()

    # Running flag
    _isRunning = False

    # Hash of meters
    _hashMeter = dict()

    # Write directory
    _writeDir = "/tmp/"

    # Write interval in millis
    _writeIntervalMs = 60000

    # Write greenlet
    _writeGreenlet = None

    # Purge on start
    _purgeOnStart = True

    # Purge on stop
    _purgeOnStop = True

    #=============================================
    # START & STOP
    #=============================================

    @classmethod
    def start(cls, writeDir, writeIntervalMs, purgeOnStart, purgeOnStop):
        """
        Start the manager
        :param writeDir: Write directory
        :type writeDir: str
        :param writeIntervalMs: Write interval in millis
        :type writeIntervalMs: int
        :param purgeOnStart: Purge files on start (based on SolBase._compoName)
        :type purgeOnStart: bool
        :param purgeOnStop: Purge files on stop (based on SolBase._compoName)
        :type purgeOnStop: bool
        :return Nothing
        """

        with cls._staticLock:
            try:
                lifecyclelogger.info("Start : starting")

                # Check
                if cls._isRunning:
                    logger.warn("Already running, doing nothing")
                elif not os.path.isdir(writeDir):
                    raise Exception("Invalid writeDir={0}".format(writeDir))

                # Store
                cls._writeDir = writeDir
                cls._writeIntervalMs = writeIntervalMs
                cls._purgeOnStart = purgeOnStart
                cls._purgeOnStop = purgeOnStop

                # Start logs
                lifecyclelogger.info("Start : _writeDir=%s", cls._writeDir)
                lifecyclelogger.info("Start : _writeIntervalMs=%s", cls._writeIntervalMs)
                lifecyclelogger.info("Start : _purgeOnStart=%s", cls._purgeOnStart)
                lifecyclelogger.info("Start : _purgeOnStop=%s", cls._purgeOnStop)

                # Purge
                if cls._purgeOnStart:
                    cls._purgeFiles()

                # Schedule next write now
                cls._writeGreenlet = gevent.spawn_later(cls._writeIntervalMs / 1000.0, cls._writeMeters)

                # Signal
                cls._isRunning = True
                lifecyclelogger.info("Start : started")
            except Exception as e:
                logger.error("e=%s", SolBase.exToStr(e))

    @classmethod
    def stop(cls):
        """
        Stop the manager
        :return  Nothing
        """

        # Signal out of lock (may help greenlet to exit itself)
        cls._isRunning = False

        with cls._staticLock:
            try:
                lifecyclelogger.info("Stop : stopping")
                # Kill the greenlet
                if cls._writeGreenlet:
                    gevent.kill(cls._writeGreenlet)
                    cls._writeGreenlet = None

                lifecyclelogger.info("Stop : stopped")
            except Exception as e:
                logger.error("e=%s", SolBase.exToStr(e))


    #=============================================
    # PURGE
    #=============================================

    @classmethod
    def _purgeFiles(cls):
        """
        Purge files
        :return Nothing
        """

        mask = "{0}{1}Meters.{2}.*.txt".format(cls._writeDir, os.sep, SolBase.getCompoName())

        logger.info("Purging files : mask=%s", mask)
        for fn in glob.glob(mask):
            logger.info("Purging files, fn=%s", fn)
            try:
                os.remove(fn)
            except Exception as e:
                logger.warn("Purging files, fn=%s, e=%s", fn, SolBase.exToStr(e))

    #=============================================
    # WRITE
    #=============================================

    @classmethod
    def _writeMeters(cls):
        """
        Write meters
        :return Nothing
        """

        with cls._staticLock:
            try:
                # Check
                if not cls._isRunning:
                    return

                # File to write to
                fName = "{0}{1}Meters.{2}.{3}.{4}.txt".format(cls._writeDir, os.sep, SolBase.getCompoName(),
                                                              os.getppid(), os.getpid())

                # Buffer to write
                buf = "D={0}\n".format(SolBase.dateCurrent())

                # Browse counters
                for k, meter in cls._hashMeter.iteritems():
                    # Go
                    try:
                        # Get it
                        d = meter.toDict()

                        # Check it
                        if not isinstance(d, dict):
                            logger.warn("toDict not returned a dict for k=%s, meter=%s, having=%s", k, meter, d)
                            continue

                        # Browse the dict
                        for name, val in d.iteritems():
                            buf += "{0}={1}\n".format(name, val)
                    except Exception as e:
                        logger.warn("Exception for k=%s, meter=%s, e=%s", k, meter, e)
                        continue

                # Write to file
                try:
                    with open(fName, "w") as f:
                        f.write(buf)
                except Exception as e:
                    logger.warn("Failed to write, fName=%s, e=%s", fName, e)

                # Schedule next write now
                if cls._isRunning:
                    cls._writeGreenlet = gevent.spawn_later(cls._writeIntervalMs / 1000.0, cls._writeMeters)
            except GreenletExit:
                pass
            except Exception as e:
                logger.error("e=%s", SolBase.exToStr(e))

    #=============================================
    # METER STUFF
    #=============================================

    @classmethod
    def put(cls, meter):
        """
        Put a new meter.
        :param meter: A MeterBase derived class instance.
        :type meter: MeterBase
        :return True if instance has been added, False otherwise (case where the class is already registered)
        :rtype bool
        """

        # Get the key
        key = meter.__class__.__name__

        # Check
        if cls._hashMeter.has_key(key):
            logger.debug("key=%s already added, skipping", key)
            return False

        # Store
        cls._hashMeter[key] = meter
        return True

    @classmethod
    def get(cls, meter):
        """
        Get the meter for specified class
        :param meter: A MeterBase derived class (not an instance)
        :type meter: MeterBase
        :return MeterBase
        :rtype MeterBase
        """

        # Get
        m = cls._hashMeter.get(meter.__name__)
        if not m:
            logger.warn("Class not registered, key=%s", meter.__name__)
            return None
        return m













