"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""

# Logger
from Queue import Empty
from socket import error
from errno import EWOULDBLOCK
import logging
from threading import Lock

from gevent import GreenletExit
import gevent
from gevent.queue import Queue
from gevent.socket import socket

from pythonsol.Meter.MeterManager import MeterManager
from pythonsol.SolBase import SolBase
from pythonsol.TcpBase.SignaledBuffer import SignaledBuffer
from pythonsol.TcpServer.TcpServerStat import TcpServerStat


SolBase.loggingInit()
logger = logging.getLogger("TcpSocketManager")


class TcpSocketManager(object):
    """
    Tcp socket manager.
    """

    # Send queue : Wait in ms per loop.
    # Upon queue fill, called is unlocked immediately.
    # We use a call go get(Block=True, QUEUE_WAIT_MS_PER_LOOP) to avoid eating cpu while waiting.
    # Value in second OR None for unlimited wait.
    QUEUE_WAIT_SEC_PER_LOOP = None

    #=====================================================
    # HELPER FOR SOCKET CLOSING
    #=====================================================

    @classmethod
    def safeCloseSocket(cls, socToClose):
        """
        Safe close a socket
        :param cls: cls
        :param socToClose: socket
        :type socToClose: socket
        :return: Nothing
        """

        if socToClose is None:
            return

        try:
            socToClose.shutdown(2)
        except Exception as e:
            logger.debug("Socket shutdown ex=%s", SolBase.exToStr(e))

        try:
            socToClose.close()
        except Exception as e:
            logger.debug("Socket close ex=%s", SolBase.exToStr(e))

        try:
            del socToClose
        except Exception as e:
            logger.debug("Socket del ex=%s", SolBase.exToStr(e))

    def __init__(self, callbackDisconnect, callbackReceive, onReceiveProfileSelf=False):
        """
        Constructor.
        :param callbackDisconnect: Callback to call upon socket disconnection.
        :param callbackReceive:  Callback to call upon socket receive.
        :param onReceiveProfileSelf: If True, self will be provided as onReceive param
        :type onReceiveProfileSelf: bool
        :return: Nothing.
        """

        # Connected flag (backed by isConnected property)
        self.isConnected = False

        # Socket instance
        self.__socket = None

        # Send queue
        self.__sendQueue = Queue()

        # Callback : called when something is received from the socket
        self._callbackReceive = callbackReceive
        self._onReceiveProfileSelf = onReceiveProfileSelf

        # Callback : called upon socket disconnection
        self._callbackDisconnect = callbackDisconnect

        # Timestamps
        self._dtCreated = SolBase.dateCurrent()
        self._dtLastRecv = self._dtCreated
        self._dtLastSend = self._dtCreated

        # SSL Handshake asynch
        self.__sslHandshakeAsynch = False
        self.__sslHandshakePending = False
        self.__sslHandshakeTimeoutMs = None
        self.__sslWaitDebugMs = None
        self.__sslTimeoutGreenlet = None
        self.__sslLocker = Lock()

        # SSL handshake time
        self.__sslHandshakeMs = None

        # Fatal error
        self.__internalFatalError = False

        # Check the TcpServerStats
        if MeterManager.get(TcpServerStat) is None:
            logger.warn("__init__ : TcpServerStat not registered in MeterManager, adding it now")
            MeterManager.put(TcpServerStat())

    #================================
    # TO STRING OVERWRITE
    #================================

    def __str__(self):
        """
        To string override
        :return: A string
        :rtype string
        """

        return "dt.creat={0}*dt.recv={1}*dt.send={2}*q.send.size={3}*is.co={4}/{5}*cl={6}*id={7}".format(
            int(SolBase.dateDiff(self._dtCreated) * 0.001),
            int(SolBase.dateDiff(self._dtLastRecv) * 0.001),
            int(SolBase.dateDiff(self._dtLastSend) * 0.001),
            self.__sendQueue.qsize(),
            self.isConnected,
            self.__isRunning(),
            self.__class__.__name__,
            id(self)
        )

    def setCallbackReceive(self, callbackReceive, provideSelf):
        """
        Set the receive callback
        :param callbackReceive: Callback
        :param provideSelf: If false, callbackReceive(localBuffer) otherwise callbackReceive(localBuffer, self)
        :type provideSelf: bool
        :return: Nothing
        """

        self._callbackReceive = callbackReceive
        self._onReceiveProfileSelf = provideSelf

    #================================
    # SSL HANDSHAKE TIMEOUT
    #================================

    def __sheduleSslHandshakeTimeOut(self):
        """
        Shedule a ssl handshake timeout
        :return: Nothing
        """

        with self.__sslLocker:
            logger.debug("__sheduleSslHandshakeTimeOut : ms=%s, self=%s", self.__sslHandshakeTimeoutMs, self)
            self.__sslTimeoutGreenlet = gevent.spawn_later(self.__sslHandshakeTimeoutMs * 0.001, self.__onSslTimeout)

    def _unscheduleSslHandshakeTimeOut(self):
        """
        Unschedule the ssl timeout
        :return: Nothing
        """
        with self.__sslLocker:
            logger.debug("_unscheduleSslHandshakeTimeOut, self=%s", self)

            if not self.__sslTimeoutGreenlet is None:
                self.__sslTimeoutGreenlet.kill()
                self.__sslTimeoutGreenlet = None

    def __onSslTimeout(self):
        """
        Called when SSL timeout.
        :return: Nothing
        """
        # Reset first (in LOCK)
        with self.__sslLocker:
            self.__sslTimeoutGreenlet = None

        # Process (outside of LOCK, high level may call _unscheduleSslHandshakeTimeOut in lock)
        logger.debug("__onSslTimeout, self=%s", self)

        # Check
        if not self.__sslHandshakePending:
            # Done, exit
            return
        elif not self.__isRunning():
            # Not connected, exit
            return

        # Not done, FATAL
        logger.warn("handshake timeout, fatal, ms=%s, self=%s", self.__sslHandshakeTimeoutMs, self)

        # Stat
        MeterManager.get(TcpServerStat).sslHandshakeTimeoutCount.increment()

        # Timeout, Fatal, exit now
        self._disconnectHelper("__onSslTimeout")

    #================================
    # GETTER/SETTER
    #================================

    def setSslHandshakeAsynch(self, sslHandshakeAsynchEnable, sslHandshakeTimeoutMs, sslWaitDebugMs=None):
        """
        Enable or disable ssl asynch handshake.
        :param sslHandshakeAsynchEnable: Boolean. If True, will enable it.
        :param sslHandshakeTimeoutMs: Apply if asynch handshake is enable. Timeout in ms for handshake itself.
        :param sslWaitDebugMs: Debug only.
        :return: Nothing.
        """

        self.__sslHandshakeAsynch = sslHandshakeAsynchEnable
        self.__sslHandshakeTimeoutMs = sslHandshakeTimeoutMs
        self.__sslWaitDebugMs = sslWaitDebugMs

        # If enable, switch to pending true
        if self.__sslHandshakeAsynch:
            # Now pending
            self.__sslHandshakePending = True

            # Schedule a greenlet, waiting for ssl to complete
            self.__sheduleSslHandshakeTimeOut()


    def _setSslHandshakeMs(self, ms):
        """
        Set the time taken to perform the ssl handshake.
        :param ms: Millis.
        :return: Nothing
        """
        self.__sslHandshakeMs = ms

    def _setSocket(self, mySocket):
        """
        Setter
        :param mySocket: The socket.
        :return Nothing.
        """
        self.__socket = mySocket

    def _setIsConnected(self, myBool):
        """
        Setter
        :param myBool: New value.
        :return Nothing.
        """
        self.__isConnected = myBool

    def _getSslHandshakeMs(self):
        """
        Return the ssl handshake elapsed ms.
        :return: Integer or None if not set.
        """
        return self.__sslHandshakeMs

    def _getSocket(self):
        """
        Getter.
        return: The current socket.
        """
        return self.__socket

    def _getIsConnected(self):
        """
        Getter.
        return: A boolean.
        """
        return self.__isConnected

    def _getSendQueue(self):
        """
        Getter.
        return: The gevent send queue.
        """
        return self.__sendQueue

    def _getDtCreated(self):
        """
        Getter.
        return: A datetime.
        """
        return self._dtCreated

    def _getDtLastSend(self):
        """
        Getter.
        return: A datetime.
        """
        return self._dtLastSend

    def _getDtLastRecv(self):
        """
        Getter.
        return: A datetime.
        """
        return self._dtLastRecv

    #================================
    # PROPERTIES
    #================================

    # Properties
    isConnected = property(_getIsConnected, _setIsConnected)
    currentSocket = property(_getSocket, _setSocket)
    sendQueue = property(_getSendQueue)

    # Properties : timestamp
    dateCreated = property(_getDtCreated)
    dateLastSend = property(_getDtLastSend)
    dateLastRecv = property(_getDtLastRecv)

    #================================
    # SEND TO SOCKET
    #================================

    def sendBinaryToSocket(self, bufferToSend):
        """
        Send to socket, asynch.
        :param bufferToSend: The localBuffer to send (a str or bytes)
        :return: True is send has been scheduled, false otherwise.
        """

        # Check
        if not isinstance(bufferToSend, str):
            logger.error("TcpSocketManager : sendBinaryToSocket : bufferToSend not a binary, class=%s, self=%s",
                         SolBase.getClassName(bufferToSend), self)
            return False

        # Check
        if not self.__isRunning():
            logger.debug("TcpSocketManager : sendBinaryToSocket : not connected, returning false, self=%s", self)
            return False

        # Enqueue
        self.__sendQueue.put(bufferToSend)

        # Stats
        MeterManager.get(TcpServerStat).serverBytesSendPending.increment(len(bufferToSend))

        return True

    def sendBinaryToSocketWithSignal(self, signaledBuffer):
        """
        Send to socket, asynch.
        Upon completion, signaledBuffer.sendEvent is set.
        Caution : Caller MUST check the boolean returned. If False, the event will NOT be set.
        :param signaledBuffer: A signaledBuffer instance.
        :type signaledBuffer: SignaledBuffer
        :return: True is send has been scheduled, false otherwise.
        """

        # Check
        if not isinstance(signaledBuffer, SignaledBuffer):
            logger.error(
                "TcpSocketManager : sendBinaryToSocketWithSignal : signaledBuffer not a SignaledBuffer, class=%s, self=%s",
                SolBase.getClassName(signaledBuffer), self)
            return False

            # Check
        if not isinstance(signaledBuffer.binaryBuffer, str):
            logger.error(
                "TcpSocketManager : sendBinaryToSocketWithSignal : binaryBuffer not a binary, class=%s, self=%s",
                SolBase.getClassName(signaledBuffer.binaryBuffer), self)
            return False

        # Check
        if not self.__isRunning():
            logger.debug("TcpSocketManager : sendBinaryToSocketWithSignal : not connected, returning false, self=%s",
                         self)
            return False

        # Enqueue
        self.__sendQueue.put(signaledBuffer)

        # Stats
        MeterManager.get(TcpServerStat).serverBytesSendPending.increment(len(signaledBuffer.binaryBuffer))

        return True

    def sendTextToSocket(self, textToSend, appendLF=True):
        """
        Send text to socket, asynch.
        :param textToSend: The text to send (str)
        :param appendLF: If true, append an \n
        :return: True is send has been scheduled, false otherwise.
        """

        # Check
        if not isinstance(textToSend, str):
            logger.error("TcpSocketManager : sendTextToSocket : textToSend not an str, class=%s, buf=%s, self=%s",
                         SolBase.getClassName(textToSend), repr(textToSend), self)
            return False

        # Enqueue
        if appendLF:
            return self.sendBinaryToSocket(textToSend + "\n")
        else:
            return self.sendBinaryToSocket(textToSend)

    def sendUnicodeToSocket(self, unicodeToSend, encoding="utf-8", appendLF=True):
        """
        Send text to socket, asynch.
        :param unicodeToSend: The text to send (unicode)
        :param encoding: The encoding to use.
        :param appendLF: If true, append an \n
        :return: True is send has been scheduled, false otherwise.
        """

        # Check
        if not isinstance(unicodeToSend, unicode):
            logger.error(
                "TcpSocketManager : sendUnicodeToSocket : unicodeToSend not an unicode, class=%s, unicode=%s, self=%s",
                SolBase.getClassName(unicodeToSend), repr(unicodeToSend), self)
            return False

        # Go
        unicodeTemp = unicodeToSend

        # LF if required
        if appendLF:
            unicodeTemp += u"\n"

        # Convert to binary localBuffer
        binBuf = SolBase.unicodeToBinary(unicodeTemp, encoding)

        # Send binary
        return self.sendBinaryToSocket(binBuf)


    def getSendQueueLen(self):
        """
        Return the send queue len.
        :return The queue length (integer)
        """
        return self.__sendQueue.qsize()

    #================================
    # DISCONNECT HELPER
    #================================

    def unsetInternalFatalErrorForReconnect(self):
        """"
        This must be called before firing a reconnect AFTER a disconnection
        """

        self.__internalFatalError = False

    def _disconnectHelper(self, reason):
        """
        Fire a disconnect on our end and notify upper level is possible
        :param reason: Reason
        :type reason: str
        """

        logger.debug("Disconnect from inside, reason=%s", reason)

        # Disengage r/w loops now
        # CAUTION : NEVER (NEVER) internally flag "isConnected=False" here, you will BLAST upper level implementations
        self.__internalFatalError = True

        # Close the socket
        TcpSocketManager.safeCloseSocket(self.currentSocket)
        self.currentSocket = None

        # Callback now
        if not self._callbackDisconnect is None:
            self._callbackDisconnect()
        else:
            logger.error(
                "_disconnectHelper is None, check your implementation, self=%s", self)

    def __isRunning(self):
        """
        Return true if we are connected and we do not have a fatal error
        :return: bool
        """

        return self.isConnected == True and self.__internalFatalError == False


    #================================
    # SOCKET WAIT
    #================================

    def _waitForSocketRecv(self):
        """
        Wait for socket available for read (recv).
        :return Return True if available, False otherwise.
        """

        # Check
        if not self.__isRunning():
            logger.debug("TcpSocketManager : _waitForSocketRecv : not connected, returning false, self=%s", self)
            return False

        # Go
        try:
            # Wait
            gevent.socket.wait_read(self.__socket.fileno())

            # Ready
            return True
        except Exception as e:
            logger.warn("TcpSocketManager : _waitForSocketRecv : Exception, ex=%s, self=%s", SolBase.exToStr(e), self)
            return False

    def _waitForSocketSend(self):
        """
        Wait for socket available for write (send).
        :return Return True if available, False otherwise.
        """

        # Check
        if not self.__isRunning():
            logger.debug("TcpSocketManager : _waitForSocketSend : not connected, returning false, self=%s", self)
            return False

        # Go
        try:
            # Wait
            gevent.socket.wait_write(self.currentSocket.fileno())

            # Ready
            return True
        except Exception as e:
            logger.warn("TcpSocketManager : _waitForSocketSend : Exception, ex=%s, self=%s", SolBase.exToStr(e), self)
            return False

    #================================
    # SOCKET LOOP : READ
    #================================

    def _readLoop(self):
        """
        High level read loop on socket
        """
        logger.debug("TcpSocketManager : _readLoop : entering now, self=%s", self)
        try:
            self._readLoopInternal()
        except GreenletExit:
            logger.debug("TcpSocketManager : _readLoop : exiting due to GreenletExit, self=%s", self)
            return
        except Exception as e:
            logger.error("TcpSocketManager : _readLoop : Exception raised, ex=%s, self=%s", SolBase.exToStr(e), self)
        finally:
            logger.debug("TcpSocketManager : _readLoop : exiting now, self=%s", self)
            SolBase.sleep(0)

    def _readLoopInternal(self):
        """
        Low level read loop on socket
        """
        logger.debug("TcpSocketManager : _readLoopInternal : entering now, self=%s", self)
        try:
            while self.__isRunning():
                try:
                    if self.__sslHandshakePending:
                        # Pending SSL handshake + received something
                        # Handle SSL handshake now

                        # Stats
                        MeterManager.get(TcpServerStat).delayServerAcceptToSslHandshakeStart.put(
                            SolBase.dateDiff(self._dtCreated))

                        # Timestamps
                        self._dtLastRecv = SolBase.dateCurrent()

                        # Debug ONLY
                        if not self.__sslWaitDebugMs is None:
                            logger.error("Debug : waiting for SSL handshake, ms=%s, self=%s", self.__sslWaitDebugMs,
                                         self)
                            SolBase.sleep(self.__sslWaitDebugMs)
                            logger.error("Debug : waiting for SSL handshake : done, self=%s", self)

                        # Do the handshake
                        self.currentSocket.do_handshake()

                        # Done, cancel timeout
                        self._unscheduleSslHandshakeTimeOut()

                        # Ms
                        ms = SolBase.dateDiff(self._dtLastRecv)

                        # SSL Stats (for client)
                        self._setSslHandshakeMs(ms)

                        # Server stats
                        MeterManager.get(TcpServerStat).delayServerSslHandshake.put(ms)

                        # Done
                        self.__sslHandshakePending = False

                        # Non blocking mode now
                        self.currentSocket.setblocking(0)
                        self.currentSocket.settimeout(None)

                        # Reloop in normal mode
                        continue

                    # Wait for socket to be available for read
                    ok = self._waitForSocketRecv()
                    if not ok:
                        # This is not really normal
                        logger.warn("TcpSocketManager : _readLoopInternal : _waitForSocketRecv returned False, self=%s",
                                    self)
                    elif not self.__isRunning():
                        logger.debug(
                            "TcpSocketManager : _readLoopInternal : _waitForSocketRecv returned True, __isRunning()==False, exiting, self=%s",
                            self)
                        return
                    else:
                        # Something to read...
                        localBuffer = self._readFromSocket()
                        if not self.__isRunning():
                            logger.debug(
                                "TcpSocketManager : _readLoopInternal : _readFromSocket returned, __isRunning()==False, exiting, self=%s",
                                self)
                        elif localBuffer is None:
                            # This is not really normal
                            logger.debug(
                                "TcpSocketManager : _readLoopInternal : _readFromSocket returned None, self=%s", self)
                        elif len(localBuffer) == 0:
                            # This is not really normal
                            logger.debug(
                                "TcpSocketManager : _readLoopInternal : _readFromSocket returned empty string, self=%s",
                                self)
                        else:
                            # Timestamps
                            self._dtLastRecv = SolBase.dateCurrent()

                            # Notify
                            if not self._callbackReceive is None:
                                # Self provide ?
                                if self._onReceiveProfileSelf:
                                    self._callbackReceive(localBuffer, self)
                                else:
                                    self._callbackReceive(localBuffer)

                            else:
                                logger.error(
                                    "TcpSocketManager : _readLoopInternal : _callbackReceive is None, check you implementation, self=%s",
                                    self)

                    # Next read
                    SolBase.sleep(0)
                except Exception as e:
                    logger.warn("TcpSocketManager : _readLoopInternal : IN_LOOP Exception raised, ex=%s, self=%s",
                                SolBase.exToStr(e), self)
        except Exception as e:
            logger.error("TcpSocketManager : _readLoopInternal : METHOD Exception raised, ex=%s, self=%s",
                         SolBase.exToStr(e), self)
        finally:
            logger.debug("TcpSocketManager : _readLoopInternal : exiting now, self=%s", self)
            SolBase.sleep(0)

    def _readFromSocket(self):
        """
        Read from the socket. Return a localBuffer or None.
        """

        # Check
        if not self.__isRunning():
            logger.debug("TcpSocketManager : _readFromSocket : not connected, doing nothing, self=%s", self)
            return None

        try:
            #TODO : Socket localBuffer size
            #TODO : Socket, upon server closure, will BLOCK on recv. Upon time, auto disconnect the client.

            # Try to read
            localBuffer = self.currentSocket.recv(1024)

            # If localBuffer is empty string, socket is disconnected.
            if not localBuffer is None and len(localBuffer) == 0:
                # Disconnect
                logger.debug(
                    "TcpSocketManager : _readFromSocket : empty string received, disconnecting ourselves, self=%s",
                    self)
                self._disconnectHelper("_readFromSocket : Empty string")
                return None

            # Stats
            MeterManager.get(TcpServerStat).serverBytesReceived.increment(len(localBuffer))
            # Ok
            return localBuffer
        except error as e:
            # [Errno 11] Resource temporarily unavailable
            # Means that nothing is available to read.
            # If not this, we raise.
            if e.args[0] != EWOULDBLOCK:
                # Raise :)
                logger.debug("_tryReadWrite : _readFromSocket : error, ex=%s, self=%s", SolBase.exToStr(e), self)
                self._disconnectHelper("_readFromSocket : error / No EWOULDBLOCK")
                return None
            else:
                # Normal
                logger.debug("_tryReadWrite : _readFromSocket : normal exception/EWOULDBLOCK, ex=%s, self=%s", e, self)
                return None
        except Exception as e:
            logger.debug("_tryReadWrite : _readFromSocket : Exception, ex=%s, self=%s", SolBase.exToStr(e), self)
            self._disconnectHelper("_readFromSocket : Exception / No EWOULDBLOCK")
            return None

    #================================
    # SOCKET LOOP : WRITE
    #================================

    def _writeLoop(self):
        """
        High level read loop on socket
        """
        logger.debug("TcpSocketManager : _writeLoop : entering now, self=%s", self)
        try:
            self._writeLoopInternal()
        except GreenletExit:
            logger.debug("TcpSocketManager : _writeLoop : exiting due to GreenletExit, self=%s", self)
            return
        except Exception as e:
            logger.error("TcpSocketManager : _writeLoop : Exception raised, ex=%s, self=%s", SolBase.exToStr(e), self)
        finally:
            logger.debug("TcpSocketManager : _writeLoop : exiting now, , self=%s", self)
            SolBase.sleep(0)

    def _writeLoopInternal(self):
        """
        Low level read/write loop on socket
        """
        logger.debug("TcpSocketManager : _writeLoopInternal : entering now, self=%s", self)
        try:
            while self.__isRunning():
                try:
                    # Wait for the queue
                    try:
                        # Call, with blocking.
                        item = self.sendQueue.get(True, TcpSocketManager.QUEUE_WAIT_SEC_PER_LOOP)

                    except Empty:
                        # Next read
                        SolBase.sleep(0)
                        continue

                    # Go some
                    if isinstance(item, str):
                        # Length
                        length = len(item)
                        # Buffer to send
                        bufToSend = item
                        # Event to signal
                        eventToSignal = None
                    elif isinstance(item, SignaledBuffer):
                        # Stats
                        length = len(item.binaryBuffer)
                        # Buffer to send
                        bufToSend = item.binaryBuffer
                        # Event to signal
                        eventToSignal = item.sendEvent
                    else:
                        logger.warn(
                            "TcpSocketManager : _writeLoopInternal : not managed class in queue, class=%s, item=%s",
                            SolBase.getClassName(item), item)
                        continue

                    # Stat now (mantis 687)
                    MeterManager.get(TcpServerStat).serverBytesSendPending.increment(-length)

                    # Wait for socket to be available for write
                    logger.debug("TcpSocketManager : _writeLoopInternal : waiting for socket to send=%s, self=%s",
                                 repr(item), self)
                    ok = self._waitForSocketSend()
                    if not ok:
                        # This is not really normal
                        logger.warn(
                            "TcpSocketManager : _writeLoopInternal : _waitForSocketSend returned False, self=%s", self)
                    elif not self.__isRunning():
                        logger.debug(
                            "TcpSocketManager : _writeLoopInternal : Ready for send, but __isRunning==False, exiting, send=%s, self=%s",
                            self)
                        return
                    else:
                        try:
                            # Ready to write, fire
                            logger.debug(
                                "TcpSocketManager : _writeLoopInternal : writing to socket to send=%s, self=%s",
                                repr(item), self)
                            ok = self._writeToSocket(bufToSend)
                            if not ok:
                                # This is not really normal
                                logger.warn(
                                    "TcpSocketManager : _writeLoopInternal : _writeToSocket returned False, self=%s",
                                    self)

                            # Signal if applicable
                            if eventToSignal:
                                eventToSignal.set()
                        finally:
                            # Timestamps
                            self._dtLastSend = SolBase.dateCurrent()

                            # Stats
                            MeterManager.get(TcpServerStat).serverBytesSendDone.increment(length)

                    # Next read
                    SolBase.sleep(0)
                except Exception as e:
                    logger.warn("TcpSocketManager : _writeLoopInternal : IN_LOOP Exception raised, ex=%s, self=%s",
                                SolBase.exToStr(e), self)
        except Exception as e:
            logger.error("TcpSocketManager : _writeLoopInternal : METHOD Exception raised, ex=%s, self=%s",
                         SolBase.exToStr(e), self)
        finally:
            logger.debug("TcpSocketManager : _writeLoopInternal : exiting now, , self=%s", self)
            SolBase.sleep(0)

    def _writeToSocket(self, localBuffer):
        """
        Write to the socket.
        """

        # Check
        if not self.__isRunning():
            logger.debug("TcpSocketManager : _writeToSocket : not connected, doing nothing, self=%s", self)
            return False

        try:
            #
            self.currentSocket.sendall(localBuffer)
            return True
        except error as e:
        # [Errno 11] Resource temporarily unavailable
        # Means that nothing is available to read.
        # If not this, we raise.
            if e.args[0] != EWOULDBLOCK:
                # Raise :)
                logger.debug("TcpSocketManager : _writeToSocket : Exception, ex=%s, self=%s", SolBase.exToStr(e), self)
                self._disconnectHelper("_writeToSocket : error / No EWOULDBLOCK")
                return False
            else:
                # Normal
                logger.debug("TcpSocketManager : _writeToSocket : normal exception/EWOULDBLOCK, ex=%s, self=%s", e,
                             self)
                return False
        except Exception as e:
            logger.debug("TcpSocketManager : _writeToSocket : Exception, ex=%s, self=%s", SolBase.exToStr(e), self)
            self._disconnectHelper("_writeToSocket : Exception")
            return False









