"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""

# Logger
import logging
import sys
from pythonsol.AtomicInt import AtomicInt
from pythonsol.DelayToCount import DelayToCount
from pythonsol.Meter.MeterBase import MeterBase
from pythonsol.SolBase import SolBase

SolBase.loggingInit()
logger = logging.getLogger("TcpServerStat")


class TcpServerStat(MeterBase):
    """
    Tcp server stat.
    """

    def __init__(self):
        """
        Constructor.
        """

        #==========================
        # GENERIC
        #==========================

        # Client connected, current
        self.clientConnected = AtomicInt()

        # Client register : count
        self.clientRegisterCount = AtomicInt()

        # Client register : exception
        self.clientRegisterException = AtomicInt()

        # Client remove : count
        self.clientRemoveCount = AtomicInt()

        # Client remove : errors, by type
        self.clientRemoveNotHashed = AtomicInt()

        # Client remove : exception
        self.clientRemoveException = AtomicInt()

        # Client remove : time out
        self.clientRemoveTimeOutInternal = AtomicInt()
        self.clientRemoveTimeOutBusiness = AtomicInt()

        #==========================
        # SEND / RECV
        #==========================

        # Server : bytes to send, pending
        self.serverBytesSendPending = AtomicInt()

        # Server : bytes sent
        self.serverBytesSendDone = AtomicInt()

        # Server : bytes received
        self.serverBytesReceived = AtomicInt()

        #==========================
        # SSL
        #==========================

        # Time from accept to ssl handshake startup
        self.delayServerAcceptToSslHandshakeStart = DelayToCount("delayServerAcceptToSslHandshakeStart")

        # Time taken by ssl handshake itself (success AND timeout)
        self.delayServerSslHandshake = DelayToCount("delayServerSslHandshake")

        # SSL handshake timeout
        self.sslHandshakeTimeoutCount = AtomicInt()

        #==========================
        # SESSIONS
        #==========================

        # Session duration in SECONDS (not in millis)
        self.sessionDurationSecond = DelayToCount("sessionDurationSecond",
                                                  delayArray=[30, 60, 300, 1800, 3600, 14400, 28800, 86400, 172800,
                                                              sys.maxint])

    def getNameValueDict(self):
        """
        Return the dict for MeterManager.
        :return: A dict
        """

        d = dict()
        d["RAW-clientConnected"] = self.clientConnected.get()
        d["RAW-clientRegisterCount"] = self.clientRegisterCount.get()
        d["RAW-clientRegisterException"] = self.clientRegisterException.get()
        d["RAW-clientRemoveCount"] = self.clientRemoveCount.get()
        d["RAW-clientRemoveNotHashed"] = self.clientRemoveNotHashed.get()
        d["RAW-clientRemoveException"] = self.clientRemoveException.get()
        d["RAW-clientRemoveTimeOutBusiness"] = self.clientRemoveTimeOutBusiness.get()
        d["RAW-clientRemoveTimeOutInternal"] = self.clientRemoveTimeOutInternal.get()
        d["RAW-serverBytesSendPending"] = self.serverBytesSendPending.get()
        d["RAW-serverBytesSendDone"] = self.serverBytesSendDone.get()
        d["RAW-serverBytesReceived"] = self.serverBytesReceived.get()
        d["RAW-sslHandshakeTimeoutCount"] = self.sslHandshakeTimeoutCount.get()
        d.update(self.delayServerAcceptToSslHandshakeStart.toDict())
        d.update(self.delayServerSslHandshake.toDict())
        d.update(self.sessionDurationSecond.toDict())
        return d

    def writeToLogger(self):
        """
        Write
        :return Nothing
        """
        logger.info("TcpServerStat : BEGIN ===")
        logger.info("TcpServerStat : clientConnected=%s", self.clientConnected.get())
        logger.info("TcpServerStat : clientRegisterCount=%s", self.clientRegisterCount.get())
        logger.info("TcpServerStat : clientRegisterException=%s", self.clientRegisterException.get())
        logger.info("TcpServerStat : clientRemoveCount=%s", self.clientRemoveCount.get())
        logger.info("TcpServerStat : clientRemoveNotHashed=%s", self.clientRemoveNotHashed.get())
        logger.info("TcpServerStat : clientRemoveException=%s", self.clientRemoveException.get())
        logger.info("TcpServerStat : clientRemoveTimeOutInternal=%s", self.clientRemoveTimeOutInternal.get())
        logger.info("TcpServerStat : clientRemoveTimeOutBusiness=%s", self.clientRemoveTimeOutBusiness.get())
        logger.info("TcpServerStat : serverBytesSendPending=%s", self.serverBytesSendPending.get())
        logger.info("TcpServerStat : serverBytesSendDone=%s", self.serverBytesSendDone.get())
        logger.info("TcpServerStat : serverBytesReceived=%s", self.serverBytesReceived.get())

        self.delayServerAcceptToSslHandshakeStart.log()
        self.delayServerSslHandshake.log()
        self.sessionDurationSecond.log()






