"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""

# Logger
import logging
from pythonsol.SolBase import SolBase

SolBase.loggingInit()
logger = logging.getLogger("TcpSimpleClient")


class TcpClientConfig(object):
    """
    Tcp client configuration.
    """

    def __init__(self):
        """
        Constructor.
        """

        # Default
        self._targetAddr = None
        self._targetPort = None

        # Ssl
        self._sslEnable = False

        # Timeout
        self._timeoutMs = 10000

        # Debug logs
        self._debugLog = False

        # Socks5 proxy
        self._proxyEnable = False
        self._proxyAddr = None
        self._proxyPort = None

        # Tcp keep alive on by default
        self._tcpKeepAliveEnabled = True
        self._tcpKeepAliveProbesSendDelayMs = 60000
        self._tcpKeepAliveProbesFailedCount = 5
        self._tcpKeepAliveProbesSendIntervalMs = 60000

    def _setTargetAddr(self, targetAddr):
        """
        Setter. Raise exception if a problem occurs.
        :param targetAddr: The target address.
        :return Nothing
        """
        if not SolBase.isStringNotEmpty(targetAddr):
            logger.error("TcpClientConfig : _setTargetAddress : not a string or empty, class=%s",
                         SolBase.getClassName(targetAddr))
            raise Exception("TcpClientConfig : _setTargetAddress : not a string or empty")
        else:
            self._targetAddr = targetAddr

    def _getTargetAddr(self):
        """
        Getter
        return: The target address.
        """
        return self._targetAddr

    def _setTargetPort(self, targetPort):
        """
        Setter. Raise exception if a problem occurs.
        :param targetPort: The target port.
        :return Nothing.
        """
        if not SolBase.isInteger(targetPort):
            logger.error("TcpClientConfig : _setTargetPort : not a int, class=%s", SolBase.getClassName(targetPort))
            raise Exception("TcpClientConfig : _setTargetPort : not a int")
        elif targetPort == 0:
            logger.warn("TcpClientConfig : _setTargetPort : newPort==0")
            raise Exception("TcpClientConfig : _setTargetPort : newPort==0")
        else:
            self._targetPort = targetPort

    def _setTimeoutMs(self, ms):
        """
        Setter. Raise exception if a problem occurs.
        :param ms: The timeout in millis or None.
        :return Nothing.
        """
        self._timeoutMs = ms

    def _getTargetPort(self):
        """
        Getter
        :return The target port.
        """
        return self._targetPort

    def _setSslEnable(self, isEnable):
        """
        Enable or disable ssl.
        :param isEnable: Boolean.
        :return: Nothing.
        """
        self._sslEnable = isEnable

    def _setDebugLog(self, isEnable):
        """
        Enable or disable debug log.
        :param isEnable: Boolean.
        :return: Nothing.
        """
        self._debugLog = isEnable


    def _getSslEnable(self):
        """
        Getter.
        :return: A boolean.
        """
        return self._sslEnable

    def _getTimeoutMs(self):
        """
        Getter.
        :return: An integer or None.
        """
        return self._timeoutMs

    def _getDebugLog(self):
        """
        Getter.
        :return: An integer or None.
        """
        return self._debugLog

    def _setProxyAddr(self, proxyAddr):
        """
        Setter. Raise exception if a problem occurs.
        :param proxyAddr: The proxy target address.
        :return Nothing
        """
        if not SolBase.isStringNotEmpty(proxyAddr):
            logger.error("TcpClientConfig : _setProxyAddress : not a string or empty, class=%s",
                         SolBase.getClassName(proxyAddr))
            raise Exception("TcpClientConfig : _setProxyAddress : not a string or empty")
        else:
            self._proxyAddr = proxyAddr

    def _getProxyAddr(self):
        """
        Getter
        return: The proxy target address.
        """
        return self._proxyAddr

    def _setProxyPort(self, proxyPort):
        """
        Setter. Raise exception if a problem occurs.
        :param proxyPort: The target proxy port.
        :return Nothing.
        """
        if not SolBase.isInteger(proxyPort):
            logger.error("TcpClientConfig : _setProxyPort : not a int, class=%s", SolBase.getClassName(proxyPort))
            raise Exception("TcpClientConfig : _setProxyPort : not a int")
        elif proxyPort == 0:
            logger.warn("TcpClientConfig : _setProxyPort : newPort==0")
            raise Exception("TcpClientConfig : _setProxyPort : newPort==0")
        else:
            self._proxyPort = proxyPort

    def _getProxyPort(self):
        """
        Getter
        :return The target port.
        """
        return self._proxyPort

    def _setProxyEnable(self, isEnable):
        """
        Enable or disable proxy.
        :param isEnable: Boolean.
        :return: Nothing.
        """
        self._proxyEnable = isEnable

    def _getProxyEnable(self):
        """
        Getter.
        :return: A boolean.
        """
        return self._proxyEnable

    #=======================
    # TCP KEEP ALIVE
    #=======================

    def _setTcpKeepAliveEnabled(self, isEnable):
        """
        Enable or disable tcp keep alive.
        :param isEnable: Boolean.
        :return: Nothing.
        """
        self._tcpKeepAliveEnabled = isEnable

    def _getTcpKeepAliveEnabled(self):
        """
        Getter.
        :return: A boolean.
        """
        return self._tcpKeepAliveEnabled


    def _setTcpKeepAliveProbesSendDelayMs(self, value):
        """
        Setter. Raise exception if a problem occurs.
        :param value: Value.
        :return Nothing.
        """
        if not SolBase.isInteger(value):
            logger.error("TcpClientConfig : _setTcpKeepAliveProbesSendDelayMs : not a int, class=%s",
                         SolBase.getClassName(value))
            raise Exception("TcpClientConfig : _setTcpKeepAliveProbesSendDelayMs : not a int")
        elif value == 0:
            logger.warn("TcpClientConfig : _setTcpKeepAliveProbesSendDelayMs : newPort==0")
            raise Exception("TcpClientConfig : _setTcpKeepAliveProbesSendDelayMs : newPort==0")
        else:
            self._tcpKeepAliveProbesSendDelayMs = value

    def _getTcpKeepAliveProbesSendDelayMs(self):
        """
        Getter
        :return Value
        :rtype int
        """
        return self._tcpKeepAliveProbesSendDelayMs

    def _setTcpKeepAliveProbesFailedCount(self, value):
        """
        Setter. Raise exception if a problem occurs.
        :param value: Value.
        :return Nothing.
        """
        if not SolBase.isInteger(value):
            logger.error("TcpClientConfig : _setTcpKeepAliveProbesFailedCount : not a int, class=%s",
                         SolBase.getClassName(value))
            raise Exception("TcpClientConfig : _setTcpKeepAliveProbesFailedCount : not a int")
        elif value == 0:
            logger.warn("TcpClientConfig : _setTcpKeepAliveProbesFailedCount : newPort==0")
            raise Exception("TcpClientConfig : _setTcpKeepAliveProbesFailedCount : newPort==0")
        else:
            self._tcpKeepAliveProbesFailedCount = value

    def _getTcpKeepAliveProbesFailedCount(self):
        """
        Getter
        :return Value
        :rtype int
        """
        return self._tcpKeepAliveProbesFailedCount

    def _setTcpKeepAliveProbesSendIntervalMs(self, value):
        """
        Setter. Raise exception if a problem occurs.
        :param value: Value.
        :return Nothing.
        """
        if not SolBase.isInteger(value):
            logger.error("TcpClientConfig : _setTcpKeepAliveProbesSendIntervalMs : not a int, class=%s",
                         SolBase.getClassName(value))
            raise Exception("TcpClientConfig : _setTcpKeepAliveProbesSendIntervalMs : not a int")
        elif value == 0:
            logger.warn("TcpClientConfig : _setTcpKeepAliveProbesSendIntervalMs : newPort==0")
            raise Exception("TcpClientConfig : _setTcpKeepAliveProbesSendIntervalMs : newPort==0")
        else:
            self._tcpKeepAliveProbesSendIntervalMs = value

    def _getTcpKeepAliveProbesSendIntervalMs(self):
        """
        Getter
        :return Value
        :rtype int
        """
        return self._tcpKeepAliveProbesSendIntervalMs

    # Properties
    targetAddr = property(_getTargetAddr, _setTargetAddr)
    targetPort = property(_getTargetPort, _setTargetPort)
    sslEnable = property(_getSslEnable, _setSslEnable)
    timeoutMs = property(_getTimeoutMs, _setTimeoutMs)
    debugLog = property(_getDebugLog, _setDebugLog)

    proxyEnable = property(_getProxyEnable, _setProxyEnable)
    proxyAddr = property(_getProxyAddr, _setProxyAddr)
    proxyPort = property(_getProxyPort, _setProxyPort)

    tcpKeepAliveEnabled = property(_getTcpKeepAliveEnabled, _setTcpKeepAliveEnabled)
    tcpKeepAliveProbesSendDelayMs = property(_getTcpKeepAliveProbesSendDelayMs, _setTcpKeepAliveProbesSendDelayMs)
    tcpKeepAliveProbesFailedCount = property(_getTcpKeepAliveProbesFailedCount, _setTcpKeepAliveProbesFailedCount)
    tcpKeepAliveProbesSendIntervalMs = property(_getTcpKeepAliveProbesSendIntervalMs,
                                                _setTcpKeepAliveProbesSendIntervalMs)





