"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
#
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""
import argparse

import os
import sys
import gevent
import atexit
import logging
import resource
import errno
import signal
import pwd
import grp
from pythonsol.SolBase import SolBase

SolBase.loggingInit()
logger = logging.getLogger("Daemon")

class Daemon(object):
    """
    Daemon helper.
    """

    def __init__(self):
         """
         Constructor
         """
         self.vars=None

    def _internalInit(self,
            pidfile,
            stdin, stdout, stderr,
            onStartExitZero,
            maxOpenFiles,
            changeDir,
            timeoutMs):
        """
        Internal init.
        :param pidfile: Pid file.
        :type pidfile: str
        :param changeDir: Enable directory change
        :type changeDir: bool
        :param maxOpenFiles: Max open files.
        :type maxOpenFiles: int
        :param stdin: stdin. What else?
        :type stdin: str
        :param stdout: stdout. What else?
        :type stdout: str
        :param stderr: stderr. What else?
        :type stderr: str
        :param onStartExitZero: perform an exit(0) on start.
        :type onStartExitZero: bool
        :param timeoutMs: Timeout in ms
        :type timeoutMs: int
        """

        # Store
        self._pidfile = pidfile
        self._maxOpenFiles = maxOpenFiles
        self._timeoutMs = timeoutMs

        self._stdin = stdin
        self._stdout = stdout
        self._stderr = stderr

        self._changeDir = changeDir
        self._onStartExitZero = onStartExitZero

        logger.info("_pidfile=%s", self._pidfile)
        logger.info("_maxOpenFiles=%s", self._maxOpenFiles)
        logger.info("_timeoutMs=%s", self._timeoutMs)

        logger.info("_stdin=%s", self._stdin)
        logger.info("_stdout=%s", self._stdout)
        logger.info("_stderr=%s", self._stderr)

        logger.info("_onStartExitZero=%s", self._onStartExitZero)
        logger.info("_changeDir=%s", self._changeDir)

        logger.info("vars=%s", self.vars)

        # Check
        if not pidfile:
            raise Exception("pidfile is required")

        # Internal
        self._pidFileHandle = None
        self._softLimit=None
        self._hardLimit=None

    #===============================================
    # UTILITIES
    #===============================================

    def _redirectAllStd(self):
        """
        Redirect std
        """

        # Init
        si=None
        so=None
        se=None

        # Flush
        logger.info("flushing")
        sys.stdout.flush()
        sys.stderr.flush()

        # Open new std
        try:
            logger.info("opening new ones")
            si = open(self._stdin, "r")
            so = open(self._stdout, "a+")
            se = open(self._stderr, "a+", 0)

            # Dup std
            logger.info("dup2 (expecting log loss now)")
            os.dup2(si.fileno(), sys.stdin.fileno())
            os.dup2(so.fileno(), sys.stdout.fileno())
            os.dup2(se.fileno(), sys.stderr.fileno())
            logger.info("dup2 done")

            # Ok
            return
        except Exception as ex:
            logger.warn("dup2 failed, fallback now, ex=%s", SolBase.exToStr(ex))
            if so:
                so.close()
            if si:
                si.close()
            if se:
                se.close()

        #-------------------------
        # FALLBACK
        #-------------------------

        si=None
        so=None
        se=None

        try:
            # Flush
            logger.info("flushing")
            sys.stdout.flush()
            sys.stderr.flush()

            # Open new std
            logger.info("opening new ones")
            si = open(self._stdin, "r")
            so = open(self._stdout, "a+")
            se = open(self._stderr, "a+", 0)

            # Dup std
            logger.info("assigning (expecting log loss now)")
            sys.stdin = si
            sys.stdout = so
            sys.stderr = se
        except Exception as ex:
            logger.error("fatal, exit(-2) now, ex=%s", SolBase.exToStr(ex))
            if so:
                so.close()
            if si:
                si.close()
            if se:
                se.close()
            sys.exit(-2)

    def _setLimits(self):
        """
        Set limits
        """

        logger.info("Setting max open file=%s", self._maxOpenFiles)
        try:
            # Get
            self._softLimit, self._hardLimit = resource.getrlimit(resource.RLIMIT_NOFILE)
            logger.info("rlimit before : soft=%s, hard=%s", self._softLimit, self._hardLimit)

            # Update
            resource.setrlimit(resource.RLIMIT_NOFILE, (self._maxOpenFiles, self._maxOpenFiles))

            # Get
            self._softLimit, self._hardLimit = resource.getrlimit(resource.RLIMIT_NOFILE)
            logger.info("rlimit after, soft=%s, hard=%s", self._softLimit, self._hardLimit)

        except Exception as ex:
            # Get
            self._softLimit, self._hardLimit = resource.getrlimit(resource.RLIMIT_NOFILE)

            # Log it
            logger.error("setrlimit failed, soft=%s, hard=%s, required=%s, ex=%s", self._softLimit, self._hardLimit, self._maxOpenFiles, SolBase.exToStr(ex))

            # This is fatal
            logger.error("failed to apply _maxOpenFiles, exit(-3) now")
            sys.exit(-3)

    def _goDaemon(self):
        """
        Daemonize us
        """

        logger.info("Entering, pid=%s", os.getpid())
        
        # Limit
        self._setLimits()


        # Fork1
        logger.info("fork1, %s", SolBase.getCurrentPidsAsString())
        try:
            pid = gevent.fork()
            if pid > 0:
                # Exit first parent
                sys.exit(0)            
        except OSError as ex:
            logger.error("fork1 failed, exit(1) now : errno=%s, err=%s, ex=%s", ex.errno, ex.strerror, SolBase.exToStr(ex))
            sys.exit(1)
        logger.info("fork1 done, %s", SolBase.getCurrentPidsAsString())

        # Diverge from parent
        if self._changeDir:
            logger.info("chdir now")
            os.chdir("/")

        # Set stuff
        logger.info("setsid and umask")
        # noinspection PyArgumentList
        os.setsid()
        os.umask(0)

        # Fork2
        logger.info("fork2, %s", SolBase.getCurrentPidsAsString())
        try:
            pid = gevent.fork()
            if pid > 0:
                # exit from second parent
                sys.exit(0)
        except OSError as ex:
            logger.error("fork2 failed, exit(2) now : errno=%s, err=%s, ex=%s", ex.errno, ex.strerror, SolBase.exToStr(ex))
            sys.exit(2)
        logger.info("fork2 done, %s", SolBase.getCurrentPidsAsString())

        # Redirect std
        self._redirectAllStd()

        # Go
        logger.info("initializing _pidfile=%s", self._pidfile)

        # Register the method called at exit
        atexit.register(self._removePidFile)

        # Write pidfile
        pid = str(os.getpid())
        try:
            f = open(self._pidfile, "w")
            f.write("%s" % pid)
            f.close()

        except IOError as ex:
            logger.info("pid file initialization failed, going exit(3), ex=%s", SolBase.exToStr(ex))
            sys.exit(3)

         # Ok
        logger.info("pid file set")

        # Finish
        logger.info("registering gevent signal handler : SIGUSR1")
        gevent.signal(signal.SIGUSR1, self._onReload)
        logger.info("registering gevent signal handler : SIGUSR2")
        gevent.signal(signal.SIGUSR2, self._onStatus)
        logger.info("registering gevent signal handler : SIGTERM")
        gevent.signal(signal.SIGTERM, self._exitHandler)

        logger.info("registering gevent signal handler : done")

        # Fatality
        SolBase.voodooInit()
        logger.info("process started, pid=%s, pidfile=%s", os.getpid(), self._pidfile)

    def _removePidFile(self):
        """
        Remove the pid file
        """
        if os.path.exists(self._pidfile):
            os.remove(self._pidfile)

    def _setUserAndGroup(self, user, group):
        """
        Set user and group
        :param user: User
        :type user: str
        :param group: Group
        :type group: str
        """
        if group:
            os.setgid(grp.getgrnam(group).gr_gid)
            logger.info("group set=%s", group)
        if user:
            os.setuid(pwd.getpwnam(user).pw_uid)
            logger.info("user set=%s", user)

    def _getRunningPid(self):
        """
        Get running pid
        :return: int
        """

        pf=None
        try:
            pf = open(self._pidfile, "r")
            return int(pf.read().strip())
        except IOError:
            return None
        finally:
            if pf:
                pf.close()

    #===============================================
    # HANDLERS
    #===============================================

    def _onStart(self):
        """
        On start
        """
        logger.info("Base implementation (pass)")

    def _onStop(self):
        """
        On stop
        """
        logger.info("Base implementation (pass)")

    #noinspection PyUnusedLocal
    def _onReload(self, *argv, **kwargs):
        """
        On reload
        """
        logger.info("Base implementation (pass)")

    #noinspection PyUnusedLocal
    def _onStatus(self, *argv, **kwargs):
        """
        On status
        """
        logger.info("Base implementation (pass)")

    # noinspection PyUnusedLocal
    def _exitHandler(self, *argv, **kwargs):
        """
        Exit handler
        """

        try:
            # Call
            self._onStop()
        finally:
            pass

        logger.info("exiting daemon with exit(0)")
        sys.exit(0)

    #===============================================
    # DAEMON METHODS
    #===============================================

    def _daemonStart(self, user, group):
        """
        Start the daemon
        :param user: User
        :type user: str
        :param group: Group
        :type group: str

        """
        """
        # Status : OK, implemented
        # - Running : exit 0 => OK
        # - Not running and pid file exist : exit 1 => OK
        # - Not running : exit 3 => OK
        # - Other : 4 => NOT TESTED
        """
        # Check for a pidfile to see if the daemon already runs
        logger.info("entering")
        pid = self._getRunningPid()

        # Pid ?
        if pid:
            # Check with SIGUSR2
            try:
                os.kill(pid, signal.SIGUSR2)

                # Check success, asked to start, but already running
                logger.info("Already running, exit(1) now, pid=%s", pid)
                sys.exit(1)
            except OSError as err:
                if err.errno == errno.ESRCH:
                    logger.info("Found pidfile but SIGUSR2 failed, pid=%s, pidfile=%s", pid, self._pidfile)
                    if os.path.exists(self._pidfile):
                        logger.info("Removing pidfile")
                        self._removePidFile()

        # Ok start now
        self._goDaemon()
        self._setUserAndGroup(user, group)
        self._onStart()

        #=====================
        # CAUTION : With same daemon, this should not happen (custom start will exit the main
        # due to unlock by customStop)
        # So, the exit(0) is USELESS and may be DISABLED
        #=====================

        # Exit
        if self._onStartExitZero is True:
            logger.info("exiting WITH exit(0) due to _onStartExitZero==True")
            sys.exit(0)
        else:
            logger.info("exiting WITHOUT exit(0)")

    def _daemonStop(self):
        """
        Stop the daemon
        # Status : OK, implemented
        # - Running : exit 0 => OK
        # - Not running and pid file exist : exit 1 => OK
        # - Not running : exit 3 => OK
        # - Other : 4 => NOT TESTED

        """

        logger.info("entering")

        # Get the pid from the pidfile
        pid = self._getRunningPid()
        if not pid:
            logger.info("daemon is not running, pidFile=%s", self._pidfile)
            return

        # Stop it
        logger.info("sending SIGTERM, pid=%s, pidFile=%s", pid, self._pidfile)
        try:
            os.kill(pid, signal.SIGTERM)
        except OSError as ex:
            if ex.errno == errno.ESRCH:
                logger.info("SIGTERM failed, ESRCH, ex=%s", SolBase.exToStr(ex))
            else:
                logger.info("SIGTERM failed, not an ESRCH, ex=%s", SolBase.exToStr(ex))
        except Exception as ex:
            logger.info("SIGTERM failed, not an OSError, going exit(1), ex=%s", SolBase.exToStr(ex))
            sys.exit(1)
        finally:
            if os.path.exists(self._pidfile):
                logger.info("Removing pidFile=%s", self._pidfile)
                self._removePidFile()

        # Ok
        logger.info("SIGTERM sent")
        msStart = SolBase.msCurrent()

        # Validate
        procTarget = "/proc/%d" % pid
        while SolBase.msDiff(msStart)<self._timeoutMs:
            if os.path.exists(procTarget):
                SolBase.sleep(100)
                continue

            # Over
            logger.info("SIGTERM success, pid=%s", pid)
            self._removePidFile()
            return

        # Not cool
        logger.warn("SIGTERM timeout=%s ms, pid=%s", self._timeoutMs, pid)    

    def _daemonStatus(self):
        """
        Check status.
        May send a SIGUSR2 to process.
        
        # Status : 
        # - Running : exit 0 
        # - Not running and pid file exist : exit 1 
        # - Not running : exit 3 
        # - Other : 4 => NOT TESTED

        """
        # Get the pid from the pidfile
        pid = self._getRunningPid()
        if not pid:
            logger.info("Daemon is not running (no pidfile), pidfile=%s", self._pidfile)
            sys.exit(3)

        # Validate
        try:
            os.kill(pid, signal.SIGUSR2)
        except OSError as err:
            if err.errno == errno.ESRCH:
                # Process not found
                logger.info("Daemon is not running (SIGUSR2 failed), pid=%s, pidfile=%s", pid, self._pidfile)
                sys.exit(1)

        # Ok
        logger.info("Daemon is running, pid=%s, pidfile=%s", pid, self._pidfile)
        sys.exit(0)

    def _daemonReload(self):
        """
        Reload.
        May send a SIGUSR1 to process.
        """

        # Get
        pid = self._getRunningPid()
        if not pid:
            logger.warn("Daemon not running, (no pidfile), pidfile=%s", self._pidfile)
            return 

        # Signal it
        try:
            os.kill(pid, signal.SIGUSR1)
        except OSError as err:
            if err.errno == errno.ESRCH:
                 # Process not found
                logger.info("Daemon is not running (SIGUSR1 failed), pid=%s, pidfile=%s", pid, self._pidfile)
                sys.exit(2)

        logger.info("Reload requested through SIGUSR1, pid=%s, pidfile=%s", pid, self._pidfile)

    #===============================================
    # COMMAND LINE PARSER
    #===============================================

    @classmethod
    def initializeArgumentsParser(cls):
        """
        Initialize the parser. 
        :param cls: class.
        :return ArgumentParser
        :rtype ArgumentParser
        """
        logger.info("Entering")

        # Create an argument parser
        argParser = argparse.ArgumentParser(description='SolBase.Daemon', add_help=True)

        # Set it
        argParser.add_argument(
            "programname",
            metavar="programname",
            type=str,
            action="store",
            help="Program name (argv[0]) [required]"
        )
        argParser.add_argument(
            "-pidfile",
            metavar="pidfile",
            type=str,
            default=None,
            action="store",
            help="pid filename [required]"
        )
        argParser.add_argument(
            "-user",
            metavar="user",
            type=str,
            default=None,
            action="store",
            help="Daemon user [optional]"
        )
        argParser.add_argument(
            "-group",
            metavar="group",
            type=str,
            default=None,
            action="store",
            help="Daemon group [optional]"
        )

        argParser.add_argument(
            "-stdin",
            metavar="stdin",
            type=str,
            default="/dev/null",
            action="store",
            help="std redirect [optional]"
        )
        argParser.add_argument(
            "-stdout",
            metavar="stdout",
            type=str,
            default="/dev/null",
            action="store",
           help="std redirect [optional]"
        )
        argParser.add_argument(
            "-stderr",
            metavar="stderr",
            type=str,
            default="/dev/null",
            action="store",
            help="std redirect [optional]"
        )
        argParser.add_argument(
            "-name",
            metavar="name",
            help="Optional name"
        )
        argParser.add_argument(
            "-maxopenfiles",
            metavar="maxopenfiles",
            type=int,
            default=1048576,
            action="store",
           help="max open files [optional]"
        )
        argParser.add_argument(
            "-timeoutms",
            metavar="timeoutms",
            type=int,
            default=15000,
            action="store",
           help="timeout when checking process [optional]"
        )
        argParser.add_argument(
            "-changedir",
            metavar="changedir",
            type=bool,
            default=False,
            action="store",
           help="if set, dir is changed after fork [optional]"
        )
        argParser.add_argument(
            "-onstartexitzero",
            metavar="onstartexitzero",
            type=bool,
            default=True,
            action="store",
           help="if set, daemon will exit zero after start [optional]"
        )
        argParser.add_argument(
            "action",
            metavar="action",
            type=str,
            choices=["start", "stop", "status", "reload"],
            action="store",
            help="Daemon action to perform (start|stop|status|reload) [required]"
        )
        logger.info("Done")
        return argParser

    @classmethod
    def parseArguments(cls, argv):
        """
        Parse command line argument (initParser required before call)
        :param cls: Our class.
        :param argv: Command line argv
        :type argv: list, tuple
        :return dict
        :rtype dict
        """

        logger.info("Entering")

        # Check argv
        if not isinstance(argv, (tuple, list)):
            raise Exception("parseArguments : argv not a list, class=%s" + SolBase.getClassName(argv))

        # Parse
        localArgs = cls.initializeArgumentsParser().parse_args(argv)

        # Flush
        d = vars(localArgs)
        logger.info("Having vars=%s", d)

        return d

    #===============================================
    # ALLOCATE (pseudo factory)
    #===============================================

    @classmethod
    def getDaemonInstance(cls):
        """
        Get a new daemon instance
        :return Daemon
        :rtype Daemon
        """
        return Daemon()

    #===============================================
    # MAIN
    #===============================================

    @classmethod
    def mainHelper(cls, argv, kwargs):
        """
        Main helper
        :param argv: Command line argv
        :type argv: list, tuple
        :param kwargs: Command line argv
        :type kwargs: dict
        :return Daemon
        :rtype Daemon
        """

        logger.info("Entering, argv=%s, kwargs=%s", argv, kwargs)

        try:
            # Parse
            varsHash = cls.parseArguments(argv)

            # Get stuff
            action=varsHash["action"]
            user=varsHash["user"]
            group=varsHash["group"]
            pidfile=varsHash["pidfile"]
            stdin=varsHash["stdin"]
            stdout=varsHash["stdout"]
            stderr=varsHash["stderr"]
            onStartExitZero=varsHash["onstartexitzero"]
            maxOpenFiles=varsHash["maxopenfiles"]
            changeDir=varsHash["changedir"]
            timeoutMs=varsHash["timeoutms"]

            # Allocate now
            logger.info("Allocating Daemon")
            di = cls.getDaemonInstance()

            # Store vars
            di.vars=varsHash

            logger.info("Internal initialization, class=%s", SolBase.getClassName(di))
            di._internalInit(
                pidfile=pidfile,
                stdin=stdin, stdout=stdout, stderr=stderr,
                onStartExitZero=onStartExitZero,
                maxOpenFiles=maxOpenFiles,
                changeDir=changeDir,
                timeoutMs=timeoutMs
            )

            logger.info("action=%s, user=%s, group=%s", action, user, group)

            if action=="start":
                di._daemonStart(user, group)
            elif action=="stop":
                di._daemonStop()
            elif action=="status":
                di._daemonStatus()
            elif action=="reload":
                di._daemonReload()
            else:
                logger.info("Invalid action=%s", action)
                print ("usage: %s -pidfile filename [_maxopenfiles int] [-timeoutms int] [-stdin string] [-stdout string] [-stderr string] [-changedir bool] [-onstartexitzero bool] [-user string] [-group string] start|stop|status|reload" % argv[0])
                sys.exit(2)

            # Done
            logger.info("Done")
            return di
        except Exception as ex:
            logger.error("Exception, ex=%s", SolBase.exToStr(ex))
            raise

#==========================
# MAIN / COMMAND LINE INTERCEPTION
#==========================

if __name__ == "__main__":
    """
    Main
    """

    try:
        # Go
        curPath = sys.path
        for s in curPath:
            logger.info("__main__ : Starting, path=%s", s)

        # Run
        Daemon.mainHelper(sys.argv, {})
    except Exception as e:
        # Failed
        logger.error("__main__ : Exception, exiting -1, ex=%s", SolBase.exToStr(e))
        sys.exit(-1)
    finally:
        logger.info("__main__ : Exiting now")

