"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
#
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""
from collections import OrderedDict
import logging
from multiprocessing import Lock
import sys
from pythonsol.AtomicInt import AtomicInt

logger = logging.getLogger("DelayToCount")


class DelayToCount(object):
    """
    A class that associate delay intervals to occurrence count.
    Low level class without any lock.
    [0-100ms  ] => x hits
    [100-200ms] => y hits
    [...-...  ] => z hits
    """

    def __init__(self, instanceName, delayArray=None):
        """
        Init
        :param delayArray: Delay array to handle in millis (must finish with max int value, value in increasing order)
        :type delayArray: list,tuple,None
        :param instanceName: Instance name
        :type instanceName: str
        :return Nothing
        """

        # Default is required
        if not delayArray:
            delayArray = [0, 50, 100, 500, 1000, 2500, 5000, 10000, 30000, 60000, sys.maxint]

        # Name
        self._instanceName = instanceName

        # Hash (ordered)
        self._sortedDict = OrderedDict()

        # Validate max int at the end (otherwise we may not be able to process and store some values)
        if delayArray[len(delayArray) - 1] != sys.maxint:
            raise Exception("sys.maxint required in last delayArray item")

        # Validate order and prepare the hash
        prev = None
        for ms in delayArray:
            if not prev:
                prev = ms
            elif prev >= ms:
                raise Exception("Not increasing value, prev={0}, ms={1}".format(prev, ms))

            # Hash it
            self._sortedDict[ms] = AtomicInt()

    def put(self, delayMs, incrementValue=1):
        """
        Put a value for specified delay.
        :param delayMs: Delay in millis
        :type delayMs: int
        :param incrementValue: Value to increment
        :type incrementValue: int
        :return Nothing
        """

        # Found the good one
        aif = self._sortedDict[sys.maxint]
        for ms, ai in self._sortedDict.iteritems():
            if ms <= delayMs and ms != sys.maxint:
                aif = ai
            else:
                break

        # Go
        aif.increment(incrementValue)

    def log(self):
        """
        Write to logger
        """

        ar = self._sortedDict.keys()
        for i in range(0, len(ar) - 1):
            ms1 = ar[i]
            ms2 = ar[i + 1]
            if ms2 == sys.maxint:
                ms2 = "MAX"
            ai = self._sortedDict[ms1]
            logger.info("DTC [%s-%s], c=%s", ms1, ms2, ai.get())

    def toDict(self):
        """
        To dict
        :return OrderedDict
        :rtype: OrderedDict
        """

        d = OrderedDict()
        ar = self._sortedDict.keys()
        for i in range(0, len(ar) - 1):
            ms1 = ar[i]
            ms2 = ar[i + 1]
            if ms2 == sys.maxint:
                ms2 = "MAX"
            ai = self._sortedDict[ms1]
            out_k = "{0}|{1}-{2}".format(self._instanceName, ms1, ms2)
            out_v = ai.get()
            d[out_k] = out_v
        return d


class DelayToCountSafe(DelayToCount):
    """
    A class that associate delay intervals to occurrence count.
    This class use lock on put.
    [0-100ms  ] => x hits
    [100-200ms] => y hits
    [...-...  ] => z hits
    """

    def __init__(self, instanceName, delayArray=None):
        """
        Init
        :param delayArray: Delay array to handle in millis (must finish with max int value, value in increasing order)
        :type delayArray: list,tuple,None
        :param instanceName: Instance name
        :type instanceName: str
        :return Nothing
        """

        self._lock = Lock()
        with self._lock:
            DelayToCount.__init__(self, instanceName, delayArray)

    def put(self, delayMs, incrementValue=1):
        """
        Put a value for specified delay.
        :param delayMs: Delay in millis
        :type delayMs: int
        :param incrementValue: Value to increment
        :type incrementValue: int
        :return Nothing
        """

        with self._lock:
            DelayToCount.put(self, delayMs, incrementValue)

    def log(self):
        """
        Write to logger
        """
        with self._lock:
            DelayToCount.log(self)

    def toDict(self):
        """
        To dict
        :return OrderedDict
        :rtype: OrderedDict
        """
        with self._lock:
            DelayToCount.toDict(self)








