"""
# -*- coding: utf-8 -*-
#===============================================================================
#
# Copyright (C) 2013/2014 Laurent Champagnac
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#===============================================================================
"""

# Logger
import logging
from gevent.queue import Queue
from pythonsol.SolBase import SolBase
from pythonsol.TcpBase.ProtocolParserTextDelimited import ProtocolParserTextDelimited
from pythonsol.TcpServer.ClientContext.TcpServerClientContext import TcpServerClientContext

SolBase.loggingInit()
logger = logging.getLogger("TcpServerQueuedClientContext")


class TcpServerQueuedClientContext(TcpServerClientContext):
    """
    Tcp server client context.
    """

    def __init__(self, tcpServer, clientId, clientSocket, clientAddr):
        """
        Constructor.
        :param tcpServer: The tcp server.
        :param clientId: The client id.
        :param clientSocket: The client socket.
        :param clientAddr: The client remote address.
        :return Nothing
        """

        # Base - we provide two callback :
        # - one for disconnecting ourselves
        # - one to notify socket receive buffer
        TcpServerClientContext.__init__(self, tcpServer, clientId, clientSocket, clientAddr)

        # Receive queue
        self.__receiveQueue = Queue()

        # Receive current buffer
        self.__receiveCurrentBuf = None

    #================================
    # TO STRING OVERWRITE
    #================================

    def __str__(self):
        """
        To string override
        :return: A string
        :rtype string
        """

        return "q.recv.size={0}*{1}".format(
            self.__receiveQueue.qsize(),
            TcpServerClientContext.__str__(self)
        )

    #===============================
    # RECEIVE
    #===============================

    def _onReceive(self, binaryBuffer):
        """
        Called on socket receive. Method parse the protocol and put receive queue.
        :param binaryBuffer: The binary buffer received.
        :return Nothing.
        """

        # Got something
        logger.debug("TcpServerQueuedClientContext : _onReceive called, binaryBuffer=%s, self=%s", repr(binaryBuffer),
                     self)

        # Parse
        self.__receiveCurrentBuf = ProtocolParserTextDelimited.parseProtocol(self.__receiveCurrentBuf,
                                                                             binaryBuffer, self.__receiveQueue, "\n")

    def getRecvQueueLen(self):
        """
        Get receive queue len.
        :return An integer.
        """
        return self.__receiveQueue.qsize()

    #=================================
    # RECEIVE QUEUE
    #=================================

    def getFromReceiveQueue(self, block=False, timeOutSec=None):
        """
        Get a buffer from the receive queue.
        :param block: If True, will block.
        :param timeOutSec: Timeout in seconds.
        :return An item queued.
        - If block is False, will return an item OR raise an Empty exception if no item.
        - If block is True AND timeOut=None, will wait forever for an item.
        - If block is True and timeOutSec>0, will wait for timeOutSec then raise Empty exception if no item.
        """

        return self.__receiveQueue.get(block, timeOutSec)
